<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\QuizAttempt;
use App\Models\QuizAttemptItem;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;

class QuizController extends Controller
{
    /**
     * Get quizzes for a course
     */
    public function getCourseQuizzes(Request $request, string $courseId): JsonResponse
    {
        try {
            $language = $request->get('language', 'en');
            
            // Verify course exists
            $course = Course::find($courseId);
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            // For MVP, we'll create a default quiz structure
            $quizzes = [[
                'id' => 'quiz_' . $courseId,
                'title' => 'Course Assessment Quiz',
                'titleSw' => 'Jaribio la Tathmini ya Kozi',
                'description' => 'Test your knowledge of the course content',
                'totalQuestions' => 10,
                'passingScore' => 70,
                'timeLimit' => 30,
                'attempts' => 3,
                'isRequired' => true,
                'orderIndex' => 1
            ]];

            return response()->json([
                'success' => true,
                'data' => [
                    'quizzes' => $quizzes
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch course quizzes',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get quiz questions
     */
    public function getQuizQuestions(Request $request, string $quizId): JsonResponse
    {
        try {
            // Extract courseId from quizId (format: quiz_courseId)
            $courseId = str_replace('quiz_', '', $quizId);
            $firebaseUid = $request->header('X-Firebase-UID');
            
            if (!$firebaseUid) {
                return response()->json([
                    'success' => false,
                    'message' => 'Firebase UID is required'
                ], 401);
            }

            $course = Course::find($courseId);
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            // Sample quiz questions - in production, these would come from database or AI generation
            $questions = [
                [
                    'id' => 'q_001',
                    'question' => 'What is the first step in starting an export business?',
                    'questionSw' => 'Ni hatua gani ya kwanza ya kuanza biashara ya uuzaji nje?',
                    'type' => 'multiple_choice',
                    'options' => [
                        [
                            'id' => 'opt_1',
                            'text' => 'Market research',
                            'textSw' => 'Utafiti wa soko'
                        ],
                        [
                            'id' => 'opt_2', 
                            'text' => 'Product selection',
                            'textSw' => 'Kuchagua bidhaa'
                        ],
                        [
                            'id' => 'opt_3',
                            'text' => 'Finding buyers',
                            'textSw' => 'Kutafuta wanunuzi'
                        ],
                        [
                            'id' => 'opt_4',
                            'text' => 'Getting licenses',
                            'textSw' => 'Kupata vibali'
                        ]
                    ]
                ],
                [
                    'id' => 'q_002',
                    'question' => 'Which document is required for export?',
                    'questionSw' => 'Hati gani inahitajika kwa uuzaji nje?',
                    'type' => 'multiple_choice',
                    'options' => [
                        [
                            'id' => 'opt_1',
                            'text' => 'Export license',
                            'textSw' => 'Leseni ya uuzaji nje'
                        ],
                        [
                            'id' => 'opt_2',
                            'text' => 'Birth certificate',
                            'textSw' => 'Cheti cha kuzaliwa'
                        ],
                        [
                            'id' => 'opt_3',
                            'text' => 'School certificate',
                            'textSw' => 'Cheti cha shule'
                        ],
                        [
                            'id' => 'opt_4',
                            'text' => 'Marriage certificate',
                            'textSw' => 'Cheti cha ndoa'
                        ]
                    ]
                ]
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'quiz' => [
                        'id' => $quizId,
                        'title' => 'Course Assessment Quiz',
                        'timeLimit' => 30,
                        'totalQuestions' => count($questions)
                    ],
                    'questions' => $questions
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch quiz questions',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Submit quiz answers
     */
    public function submitQuiz(Request $request, string $quizId): JsonResponse
    {
        try {
            $firebaseUid = $request->header('X-Firebase-UID');
            
            if (!$firebaseUid) {
                return response()->json([
                    'success' => false,
                    'message' => 'Firebase UID is required'
                ], 401);
            }

            $validated = $request->validate([
                'answers' => 'required|array',
                'answers.*.questionId' => 'required|string',
                'answers.*.selectedOptionId' => 'required|string',
                'timeSpent' => 'required|integer|min:1'
            ]);

            $courseId = str_replace('quiz_', '', $quizId);
            $answers = $validated['answers'];
            $timeSpent = $validated['timeSpent'];

            // Correct answers (in production, these would be stored in database)
            $correctAnswers = [
                'q_001' => 'opt_1', // Market research
                'q_002' => 'opt_1'  // Export license
            ];

            $correctCount = 0;
            $breakdown = [];

            foreach ($answers as $answer) {
                $isCorrect = ($correctAnswers[$answer['questionId']] ?? null) === $answer['selectedOptionId'];
                if ($isCorrect) {
                    $correctCount++;
                }

                $breakdown[] = [
                    'questionId' => $answer['questionId'],
                    'correct' => $isCorrect,
                    'selectedAnswer' => $answer['selectedOptionId'],
                    'correctAnswer' => $correctAnswers[$answer['questionId']] ?? null
                ];
            }

            $totalQuestions = count($answers);
            $score = round(($correctCount / $totalQuestions) * 100);
            $passed = $score >= 70;

            // Save quiz attempt
            $attempt = QuizAttempt::create([
                'user_firebase_uid' => $firebaseUid,
                'course_id' => $courseId,
                'quiz_id' => $quizId,
                'session_token' => 'mobile_' . md5($firebaseUid . $quizId . time()),
                'score' => $score,
                'total_questions' => $totalQuestions,
                'correct_answers' => $correctCount,
                'time_spent_seconds' => $timeSpent * 60,
                'passed' => $passed,
                'completed_at' => now()
            ]);

            // Save individual answers
            foreach ($breakdown as $item) {
                QuizAttemptItem::create([
                    'quiz_attempt_id' => $attempt->id,
                    'question_id' => $item['questionId'],
                    'selected_option_id' => $item['selectedAnswer'],
                    'is_correct' => $item['correct']
                ]);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'result' => [
                        'attemptId' => $attempt->id,
                        'score' => $score,
                        'totalQuestions' => $totalQuestions,
                        'correctAnswers' => $correctCount,
                        'passed' => $passed,
                        'passingScore' => 70,
                        'timeSpent' => $timeSpent,
                        'completedAt' => $attempt->completed_at->toISOString(),
                        'breakdown' => $breakdown
                    ]
                ]
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to submit quiz',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get user quiz results
     */
    public function getUserQuizResults(Request $request, string $firebaseUid, string $quizId): JsonResponse
    {
        try {
            $attempts = QuizAttempt::where('user_firebase_uid', $firebaseUid)
                ->where('quiz_id', $quizId)
                ->orderBy('completed_at', 'desc')
                ->get()
                ->map(function($attempt) {
                    return [
                        'attemptId' => $attempt->id,
                        'score' => $attempt->score,
                        'passed' => $attempt->passed,
                        'completedAt' => $attempt->completed_at->toISOString(),
                        'timeSpent' => round($attempt->time_spent_seconds / 60) // Convert to minutes
                    ];
                });

            $bestScore = $attempts->max('score') ?? 0;
            $attemptsUsed = $attempts->count();
            $maxAttempts = 3;

            return response()->json([
                'success' => true,
                'data' => [
                    'attempts' => $attempts,
                    'bestScore' => $bestScore,
                    'attemptsUsed' => $attemptsUsed,
                    'maxAttempts' => $maxAttempts,
                    'canRetake' => $attemptsUsed < $maxAttempts
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch quiz results',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }
}