<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\DB;

class EnrollmentController extends Controller
{
    /**
     * Enroll user in a course
     */
    public function enrollCourse(Request $request, string $courseId): JsonResponse
    {
        try {
            $firebaseUid = $request->header('X-Firebase-UID');
            
            if (!$firebaseUid) {
                return response()->json([
                    'success' => false,
                    'message' => 'Firebase UID is required'
                ], 401);
            }

            $course = Course::find($courseId);
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $user = User::where('firebase_uid', $firebaseUid)->first();
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found. Please sync your account first.'
                ], 404);
            }

            // Check if already enrolled
            $existingEnrollment = DB::table('enrollments')
                ->where('user_id', $user->id)
                ->where('course_id', $courseId)
                ->first();

            if ($existingEnrollment) {
                return response()->json([
                    'success' => false,
                    'message' => 'User is already enrolled in this course'
                ], 400);
            }

            // Create enrollment
            $enrollmentId = DB::table('enrollments')->insertGetId([
                'user_id' => $user->id,
                'course_id' => $courseId,
                'enrollment_date' => now(),
                'status' => 'active',
                'progress_percentage' => 0,
                'created_at' => now(),
                'updated_at' => now()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Successfully enrolled in course',
                'data' => [
                    'enrollmentId' => $enrollmentId,
                    'courseId' => $courseId,
                    'enrolledAt' => now()->toISOString()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to enroll in course',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get user's enrolled courses
     */
    public function getEnrolledCourses(Request $request, string $firebaseUid): JsonResponse
    {
        try {
            $user = User::where('firebase_uid', $firebaseUid)->first();
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            $enrolledCourses = DB::table('enrollments')
                ->join('courses', 'enrollments.course_id', '=', 'courses.id')
                ->where('enrollments.user_id', $user->id)
                ->where('enrollments.status', 'active')
                ->select([
                    'courses.id as courseId',
                    'courses.title',
                    'courses.image_url as imageUrl',
                    'enrollments.progress_percentage as progressPercentage',
                    'courses.total_lessons as totalLessons',
                    'enrollments.status',
                    'enrollments.enrollment_date as enrolledAt'
                ])
                ->get()
                ->map(function($enrollment) {
                    return [
                        'courseId' => $enrollment->courseId,
                        'title' => $enrollment->title,
                        'imageUrl' => $enrollment->imageUrl,
                        'progress' => [
                            'percentage' => $enrollment->progressPercentage ?? 0,
                            'totalLessons' => $enrollment->totalLessons ?? 0,
                        ],
                        'status' => $enrollment->status,
                        'enrolledAt' => $enrollment->enrolledAt
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'enrolledCourses' => $enrolledCourses
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch enrolled courses',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Update lesson progress
     */
    public function updateProgress(Request $request, string $courseId, string $lessonId): JsonResponse
    {
        try {
            $firebaseUid = $request->header('X-Firebase-UID');
            
            if (!$firebaseUid) {
                return response()->json([
                    'success' => false,
                    'message' => 'Firebase UID is required'
                ], 401);
            }

            $validated = $request->validate([
                'watchTime' => 'required|integer|min:0',
                'completed' => 'required|boolean',
                'lastPosition' => 'nullable|integer|min:0'
            ]);

            $user = User::where('firebase_uid', $firebaseUid)->first();
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            // Get enrollment
            $enrollment = DB::table('enrollments')
                ->where('user_id', $user->id)
                ->where('course_id', $courseId)
                ->first();

            if (!$enrollment) {
                return response()->json([
                    'success' => false,
                    'message' => 'User is not enrolled in this course'
                ], 400);
            }

            // Update or create lesson progress
            DB::table('lesson_progress')->updateOrInsert(
                [
                    'user_id' => $user->id,
                    'course_id' => $courseId,
                    'lesson_id' => $lessonId
                ],
                [
                    'watch_time' => $validated['watchTime'],
                    'completed' => $validated['completed'],
                    'last_position' => $validated['lastPosition'] ?? 0,
                    'updated_at' => now()
                ]
            );

            // Calculate course progress
            $totalLessons = DB::table('course_modules')
                ->where('course_id', $courseId)
                ->where('is_active', true)
                ->count();

            $completedLessons = DB::table('lesson_progress')
                ->where('user_id', $user->id)
                ->where('course_id', $courseId)
                ->where('completed', true)
                ->count();

            $progressPercentage = $totalLessons > 0 ? round(($completedLessons / $totalLessons) * 100) : 0;

            // Update enrollment progress
            DB::table('enrollments')
                ->where('user_id', $user->id)
                ->where('course_id', $courseId)
                ->update([
                    'progress_percentage' => $progressPercentage,
                    'completed_lessons' => $completedLessons,
                    'last_accessed_at' => now(),
                    'updated_at' => now()
                ]);

            // Get next lesson
            $nextLesson = DB::table('course_modules')
                ->where('course_id', $courseId)
                ->where('is_active', true)
                ->where('order_index', '>', function($query) use ($courseId, $lessonId) {
                    $query->select('order_index')
                        ->from('course_modules')
                        ->where('id', $lessonId)
                        ->limit(1);
                })
                ->orderBy('order_index')
                ->first();

            return response()->json([
                'success' => true,
                'message' => 'Progress updated',
                'data' => [
                    'lessonCompleted' => $validated['completed'],
                    'courseProgress' => $progressPercentage,
                    'nextLessonId' => $nextLesson ? $nextLesson->id : null
                ]
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update progress',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get user's shopping cart
     */
    public function getCart(Request $request, string $firebaseUid): JsonResponse
    {
        try {
            $user = User::where('firebase_uid', $firebaseUid)->first();
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            $cartItems = DB::table('cart_items')
                ->join('courses', 'cart_items.course_id', '=', 'courses.id')
                ->where('cart_items.user_id', $user->id)
                ->select([
                    'courses.id as courseId',
                    'courses.title',
                    'courses.price',
                    'courses.currency',
                    'courses.image_url as imageUrl',
                    'cart_items.created_at as addedAt'
                ])
                ->get()
                ->map(function($item) {
                    return [
                        'courseId' => $item->courseId,
                        'title' => $item->title,
                        'price' => $item->price,
                        'currency' => $item->currency,
                        'imageUrl' => $item->imageUrl,
                        'addedAt' => $item->addedAt
                    ];
                });

            $total = $cartItems->sum('price');
            $currency = $cartItems->first()->currency ?? 'TZS';

            return response()->json([
                'success' => true,
                'data' => [
                    'items' => $cartItems,
                    'total' => $total,
                    'currency' => $currency,
                    'itemCount' => $cartItems->count()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch cart',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Add course to cart
     */
    public function addToCart(Request $request, string $firebaseUid): JsonResponse
    {
        try {
            $validated = $request->validate([
                'courseId' => 'required|string|exists:courses,id'
            ]);

            $user = User::where('firebase_uid', $firebaseUid)->first();
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            // Check if already in cart
            $existingItem = DB::table('cart_items')
                ->where('user_id', $user->id)
                ->where('course_id', $validated['courseId'])
                ->first();

            if ($existingItem) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course is already in cart'
                ], 400);
            }

            DB::table('cart_items')->insert([
                'user_id' => $user->id,
                'course_id' => $validated['courseId'],
                'created_at' => now(),
                'updated_at' => now()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Course added to cart'
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to add to cart',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Remove course from cart
     */
    public function removeFromCart(Request $request, string $firebaseUid, string $courseId): JsonResponse
    {
        try {
            $user = User::where('firebase_uid', $firebaseUid)->first();
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            DB::table('cart_items')
                ->where('user_id', $user->id)
                ->where('course_id', $courseId)
                ->delete();

            return response()->json([
                'success' => true,
                'message' => 'Course removed from cart'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to remove from cart',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get user's certificates
     */
    public function getCertificates(Request $request, string $firebaseUid): JsonResponse
    {
        try {
            $user = User::where('firebase_uid', $firebaseUid)->first();
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            $certificates = DB::table('certificates')
                ->join('courses', 'certificates.course_id', '=', 'courses.id')
                ->where('certificates.user_id', $user->id)
                ->select([
                    'certificates.id',
                    'certificates.course_id as courseId',
                    'courses.title as courseTitle',
                    'certificates.issued_at as issuedAt',
                    'certificates.certificate_url as certificateUrl',
                    'certificates.verification_code as verificationCode'
                ])
                ->get()
                ->map(function($cert) {
                    return [
                        'id' => $cert->id,
                        'courseId' => $cert->courseId,
                        'courseTitle' => $cert->courseTitle,
                        'issuedAt' => $cert->issuedAt,
                        'certificateUrl' => $cert->certificateUrl,
                        'verificationCode' => $cert->verificationCode
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'certificates' => $certificates
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch certificates',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Generate certificate
     */
    public function generateCertificate(Request $request): JsonResponse
    {
        try {
            $firebaseUid = $request->header('X-Firebase-UID');
            
            if (!$firebaseUid) {
                return response()->json([
                    'success' => false,
                    'message' => 'Firebase UID is required'
                ], 401);
            }

            $validated = $request->validate([
                'courseId' => 'required|string|exists:courses,id'
            ]);

            $user = User::where('firebase_uid', $firebaseUid)->first();
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            // Check if course is completed
            $enrollment = DB::table('enrollments')
                ->where('user_id', $user->id)
                ->where('course_id', $validated['courseId'])
                ->where('progress_percentage', 100)
                ->first();

            if (!$enrollment) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course must be completed to generate certificate'
                ], 400);
            }

            // Check if certificate already exists
            $existingCert = DB::table('certificates')
                ->where('user_id', $user->id)
                ->where('course_id', $validated['courseId'])
                ->first();

            if ($existingCert) {
                return response()->json([
                    'success' => false,
                    'message' => 'Certificate already exists for this course'
                ], 400);
            }

            $verificationCode = 'EP-CERT-' . date('Y') . '-' . strtoupper(substr(md5($user->id . $validated['courseId']), 0, 6));
            $certificateUrl = 'https://example.com/certificates/' . $verificationCode . '.pdf';

            $certificateId = DB::table('certificates')->insertGetId([
                'user_id' => $user->id,
                'course_id' => $validated['courseId'],
                'certificate_url' => $certificateUrl,
                'verification_code' => $verificationCode,
                'issued_at' => now(),
                'created_at' => now(),
                'updated_at' => now()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Certificate generated successfully',
                'data' => [
                    'certificateId' => $certificateId,
                    'certificateUrl' => $certificateUrl,
                    'verificationCode' => $verificationCode
                ]
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to generate certificate',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Track analytics event
     */
    public function trackEvent(Request $request): JsonResponse
    {
        try {
            $firebaseUid = $request->header('X-Firebase-UID');
            
            $validated = $request->validate([
                'eventType' => 'required|in:course_view,lesson_start,lesson_complete,quiz_attempt',
                'courseId' => 'nullable|string|exists:courses,id',
                'lessonId' => 'nullable|string',
                'metadata' => 'nullable|array'
            ]);

            // For MVP, we'll just log the event (in production, save to analytics table)
            \Log::info('Analytics Event', [
                'user_firebase_uid' => $firebaseUid,
                'event_type' => $validated['eventType'],
                'course_id' => $validated['courseId'] ?? null,
                'lesson_id' => $validated['lessonId'] ?? null,
                'metadata' => $validated['metadata'] ?? [],
                'timestamp' => now()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Event tracked successfully'
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to track event',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }
}