<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\Category;
use App\Models\CourseReview;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;

class CourseController extends Controller
{
    /**
     * Get course details by ID
     */
    public function getCourseDetails(string $courseId): JsonResponse
    {
        try {
            // Fetch real course data from database with modules and reviews
            $course = Course::with([
                'category', 
                'subcategory', 
                'modules' => function($query) {
                    $query->where('is_active', true)->orderBy('order_index');
                },
                'reviews' => function($query) {
                    $query->approved()->with('user:id,name')->latest()->limit(5);
                }
            ])->find($courseId);

            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $courseData = [
                'id' => $course->id,
                'title' => $course->title,
                'titleSw' => $course->title_sw,
                'description' => $course->description,
                'descriptionSw' => $course->description_sw,
                'category' => $course->category ? [
                    'id' => $course->category->id,
                    'name' => $course->category->name,
                    'nameSw' => $course->category->name_sw ?? null
                ] : null,
                'subcategory' => $course->subcategory ? [
                    'id' => $course->subcategory->id,
                    'name' => $course->subcategory->name,
                    'nameSw' => $course->subcategory->name_sw ?? null
                ] : null,
                'price' => $course->price,
                'currency' => $course->currency,
                'originalPrice' => $course->original_price,
                'discount' => $course->discount_percentage,
                'duration' => $course->duration_weeks ? $course->duration_weeks . ' weeks' : null,
                'totalLessons' => $course->total_lessons,
                'totalHours' => $course->total_hours,
                'level' => $course->level,
                'language' => $course->language,
                'rating' => $course->rating_average,
                'reviewCount' => $course->review_count,
                'enrollmentCount' => $course->enrollment_count,
                'completionRate' => $course->completion_rate,
                'imageUrl' => $course->image_url,
                'previewVideoUrl' => $course->preview_video_url,
                'youtubeUrl' => $course->youtube_url,
                'coursePdfPath' => $course->course_pdf_path,
                'coursePdfName' => $course->course_pdf_name,
                'isFeatured' => $course->is_featured,
                'isActive' => $course->is_active,
                'tags' => $course->tags ?? [],
                'requirements' => $course->requirements ?? [],
                'whatYouWillLearn' => $course->what_you_will_learn ?? [],
                'certificateOffered' => $course->certificate_offered,
                'hasQuizzes' => $course->has_quizzes,
                'hasAssignments' => $course->has_assignments,
                'hasDownloadableResources' => $course->has_downloadable_resources,
                'modules' => $course->modules->map(function($module) {
                    return [
                        'id' => $module->id,
                        'title' => $module->title,
                        'titleSw' => $module->title_sw,
                        'description' => $module->description,
                        'descriptionSw' => $module->description_sw,
                        'orderIndex' => $module->order_index,
                        'durationMinutes' => $module->duration_minutes,
                        'videoUrl' => $module->video_url,
                        'youtubeUrl' => $module->youtube_url,
                        'pdfFilePath' => $module->pdf_file_path,
                        'pdfFileName' => $module->pdf_file_name,
                        'isFreePreview' => $module->is_free_preview,
                        'learningObjectives' => $module->learning_objectives,
                        'resources' => $module->resources
                    ];
                }),
                'reviews' => [
                    'recent' => $course->reviews->map(function($review) {
                        return [
                            'id' => $review->id,
                            'rating' => $review->rating,
                            'review' => $review->review,
                            'user' => [
                                'id' => $review->user->id ?? null,
                                'name' => $review->user->name ?? 'Anonymous',
                            ],
                            'is_featured' => $review->is_featured,
                            'created_at' => $review->created_at->toISOString()
                        ];
                    }),
                    'total_count' => $course->review_count,
                    'average_rating' => $course->rating_average
                ],
                'createdAt' => $course->created_at->toISOString(),
                'updatedAt' => $course->updated_at->toISOString()
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'course' => $courseData
                ]
            ]);

            // Old mock course data - keeping as fallback comment
            $course = [
                'id' => $courseId,
                'title' => 'Export Business Fundamentals',
                'titleSw' => 'Misingi ya Biashara ya Uuzaji Nje',
                'description' => 'Comprehensive course covering all aspects of export business operations',
                'descriptionSw' => 'Kozi kamili inayoshughulikia mambo yote ya shughuli za biashara ya uuzaji nje',
                'category' => [
                    'id' => 'cat_uuzaji_nje',
                    'name' => 'Export Business',
                    'nameSw' => 'Biashara ya Uuzaji Nje'
                ],
                'instructor' => [
                    'id' => 'instructor_1',
                    'name' => 'Sekela Mwambuli',
                    'title' => 'International Business Consultant',
                    'titleSw' => 'Mshauri wa Biashara ya Kimataifa',
                    'imageUrl' => 'https://ui-avatars.com/api/?name=Sekela+Mwambuli&size=200&background=7C3AED&color=fff',
                    'bio' => 'Expert in international trade with 15+ years experience',
                    'bioSw' => 'Mtaalamu wa biashara ya kimataifa mwenye uzoefu wa miaka 15+'
                ],
                'price' => 50000,
                'currency' => 'TZS',
                'originalPrice' => 75000,
                'discount' => 33,
                'duration' => '4 weeks',
                'durationSw' => 'Wiki 4',
                'totalLessons' => 12,
                'totalHours' => 24,
                'level' => 'Beginner',
                'levelSw' => 'Mwanzo',
                'language' => 'Swahili',
                'rating' => 4.8,
                'reviewCount' => 156,
                'enrollmentCount' => 342,
                'completionRate' => 87,
                'imageUrl' => 'https://images.unsplash.com/photo-1553729459-efe14ef6055d?ixlib=rb-4.0.3&auto=format&fit=crop&w=800&q=80',
                'previewVideoUrl' => null,
                'isFeatured' => true,
                'isActive' => true,
                'tags' => ['Export', 'Business', 'International Trade', 'Entrepreneurship'],
                'tagsSw' => ['Uuzaji Nje', 'Biashara', 'Biashara ya Kimataifa', 'Ujasiriamali'],
                'requirements' => [
                    'Basic understanding of business concepts',
                    'Access to internet for online materials',
                    'Commitment to complete assignments'
                ],
                'requirementsSw' => [
                    'Uelewa wa kimsingi wa dhana za kibiashara',
                    'Upatikanaji wa intaneti kwa vifaa vya mtandaoni',
                    'Kujitolea kukamilisha kazi'
                ],
                'whatYouWillLearn' => [
                    'Export documentation and procedures',
                    'International market research',
                    'Product certification and standards',
                    'Shipping and logistics management',
                    'Payment methods and financing'
                ],
                'whatYouWillLearnSw' => [
                    'Nyaraka na taratibu za uuzaji nje',
                    'Utafiti wa masoko ya kimataifa',
                    'Uthibitisho wa bidhaa na viwango',
                    'Usimamizi wa usafirishaji na lojistiki',
                    'Mbinu za malipo na ufadhili'
                ],
                'certificateOffered' => true,
                'hasQuizzes' => true,
                'hasAssignments' => true,
                'hasDownloadableResources' => true,
                'createdAt' => '2025-01-15T10:00:00Z',
                'updatedAt' => '2025-01-20T15:30:00Z'
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'course' => $course
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch course details',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get course reviews
     */
    public function getCourseReviews(Request $request, string $courseId): JsonResponse
    {
        try {
            $page = $request->get('page', 1);
            $limit = min($request->get('limit', 10), 50);
            $language = $request->get('language', 'en');
            $sortBy = $request->get('sort_by', 'created_at');
            $sortOrder = $request->get('sort_order', 'desc');
            
            // Verify course exists
            $course = Course::find($courseId);
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $query = CourseReview::with(['user:id,name,email'])
                ->where('course_id', $courseId)
                ->approved();

            // Apply sorting
            if (in_array($sortBy, ['created_at', 'rating'])) {
                $query->orderBy($sortBy, $sortOrder);
            }

            // Get total count
            $totalReviews = $query->count();

            // Apply pagination
            $reviews = $query->skip(($page - 1) * $limit)
                           ->take($limit)
                           ->get()
                           ->map(function ($review) use ($language) {
                               return [
                                   'id' => $review->id,
                                   'rating' => $review->rating,
                                   'review' => $review->getLocalizedReview($language),
                                   'user' => [
                                       'id' => $review->user->id ?? null,
                                       'name' => $review->user->name ?? 'Anonymous',
                                   ],
                                   'is_featured' => $review->is_featured,
                                   'created_at' => $review->created_at->toISOString()
                               ];
                           });

            // Calculate rating breakdown
            $ratingBreakdown = CourseReview::where('course_id', $courseId)
                ->approved()
                ->selectRaw('rating, COUNT(*) as count')
                ->groupBy('rating')
                ->pluck('count', 'rating')
                ->toArray();

            // Ensure all ratings 1-5 are present
            $fullBreakdown = [];
            for ($i = 5; $i >= 1; $i--) {
                $fullBreakdown[$i] = $ratingBreakdown[$i] ?? 0;
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'reviews' => $reviews,
                    'pagination' => [
                        'current_page' => $page,
                        'per_page' => $limit,
                        'total' => $totalReviews,
                        'total_pages' => ceil($totalReviews / $limit)
                    ],
                    'summary' => [
                        'averageRating' => $course->rating_average ?: 0,
                        'totalReviews' => $totalReviews,
                        'ratingBreakdown' => $fullBreakdown
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch course reviews',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get courses by category
     */
    public function getCoursesByCategory(Request $request, string $categoryId): JsonResponse
    {
        try {
            $language = $request->get('language', 'en');
            $page = $request->get('page', 1);
            $limit = $request->get('limit', 12);
            $sortBy = $request->get('sort_by', 'created_at');
            $sortOrder = $request->get('sort_order', 'desc');

            $query = Course::with(['category', 'subcategory'])
                ->where('category_id', $categoryId)
                ->where('is_active', true);

            // Apply sorting
            $query->orderBy($sortBy, $sortOrder);

            // Get total count before pagination
            $totalCourses = $query->count();

            // Apply pagination
            $courses = $query->skip(($page - 1) * $limit)
                           ->take($limit)
                           ->get()
                           ->map(function ($course) use ($language) {
                               return [
                                   'id' => $course->id,
                                   'title' => $course->getLocalizedTitle($language),
                                   'description' => $course->getLocalizedDescription($language),
                                   'thumbnail_url' => $course->thumbnail_url,
                                   'price' => $course->price,
                                   'original_price' => $course->original_price,
                                   'currency' => $course->currency,
                                   'discount_percentage' => $course->discount_percentage,
                                   'duration' => $course->duration,
                                   'difficulty_level' => $course->difficulty_level,
                                   'language' => $course->language,
                                   'rating' => $course->rating,
                                   'review_count' => $course->review_count,
                                   'enrollment_count' => $course->enrollment_count,
                                   'category' => [
                                       'id' => $course->category->id,
                                       'name' => $course->category->getLocalizedName($language)
                                   ],
                                   'subcategory' => $course->subcategory ? [
                                       'id' => $course->subcategory->id,
                                       'name' => $course->subcategory->getLocalizedName($language)
                                   ] : null,
                                   'is_featured' => $course->is_featured,
                                   'certificate_offered' => $course->certificate_offered,
                                   'has_quizzes' => $course->has_quizzes,
                                   'has_assignments' => $course->has_assignments,
                                   'has_downloadable_resources' => $course->has_downloadable_resources,
                                   'created_at' => $course->created_at->toISOString()
                               ];
                           });

            return response()->json([
                'success' => true,
                'data' => [
                    'courses' => $courses,
                    'category' => Category::find($categoryId),
                    'pagination' => [
                        'current_page' => $page,
                        'total_pages' => ceil($totalCourses / $limit),
                        'total_items' => $totalCourses,
                        'items_per_page' => $limit,
                        'has_next' => $page < ceil($totalCourses / $limit),
                        'has_prev' => $page > 1
                    ]
                ],
                'message' => 'Courses retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch courses by category',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get courses by subcategory
     */
    public function getCoursesBySubcategory(Request $request, string $subcategoryId): JsonResponse
    {
        try {
            $language = $request->get('language', 'en');
            $page = $request->get('page', 1);
            $limit = $request->get('limit', 12);
            $sortBy = $request->get('sort_by', 'created_at');
            $sortOrder = $request->get('sort_order', 'desc');

            $query = Course::with(['category', 'subcategory'])
                ->where('subcategory_id', $subcategoryId)
                ->where('is_active', true);

            // Apply sorting
            $query->orderBy($sortBy, $sortOrder);

            // Get total count before pagination
            $totalCourses = $query->count();

            // Apply pagination
            $courses = $query->skip(($page - 1) * $limit)
                           ->take($limit)
                           ->get()
                           ->map(function ($course) use ($language) {
                               return [
                                   'id' => $course->id,
                                   'title' => $course->getLocalizedTitle($language),
                                   'description' => $course->getLocalizedDescription($language),
                                   'thumbnail_url' => $course->thumbnail_url,
                                   'price' => $course->price,
                                   'original_price' => $course->original_price,
                                   'currency' => $course->currency,
                                   'discount_percentage' => $course->discount_percentage,
                                   'duration' => $course->duration,
                                   'difficulty_level' => $course->difficulty_level,
                                   'language' => $course->language,
                                   'rating' => $course->rating,
                                   'review_count' => $course->review_count,
                                   'enrollment_count' => $course->enrollment_count,
                                   'category' => [
                                       'id' => $course->category->id,
                                       'name' => $course->category->getLocalizedName($language)
                                   ],
                                   'subcategory' => [
                                       'id' => $course->subcategory->id,
                                       'name' => $course->subcategory->getLocalizedName($language)
                                   ],
                                   'is_featured' => $course->is_featured,
                                   'certificate_offered' => $course->certificate_offered,
                                   'has_quizzes' => $course->has_quizzes,
                                   'has_assignments' => $course->has_assignments,
                                   'has_downloadable_resources' => $course->has_downloadable_resources,
                                   'created_at' => $course->created_at->toISOString()
                               ];
                           });

            return response()->json([
                'success' => true,
                'data' => [
                    'courses' => $courses,
                    'subcategory' => \App\Models\Subcategory::with('category')->find($subcategoryId),
                    'pagination' => [
                        'current_page' => $page,
                        'total_pages' => ceil($totalCourses / $limit),
                        'total_items' => $totalCourses,
                        'items_per_page' => $limit,
                        'has_next' => $page < ceil($totalCourses / $limit),
                        'has_prev' => $page > 1
                    ]
                ],
                'message' => 'Courses retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch courses by subcategory',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Search courses with filters
     */
    public function searchCourses(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'q' => 'nullable|string|max:255',
                'category_id' => 'nullable|string|exists:categories,id',
                'subcategory_id' => 'nullable|string|exists:subcategories,id',
                'price_min' => 'nullable|numeric|min:0',
                'price_max' => 'nullable|numeric|min:0',
                'level' => 'nullable|in:beginner,intermediate,advanced',
                'duration' => 'nullable|in:short,medium,long',
                'price_type' => 'nullable|in:free,paid',
                'rating' => 'nullable|numeric|min:1|max:5',
                'language' => 'nullable|in:English,Swahili,Both',
                'sort_by' => 'nullable|in:price,rating,enrollment_count,created_at',
                'sort_order' => 'nullable|in:asc,desc',
                'page' => 'nullable|integer|min:1',
                'limit' => 'nullable|integer|min:1|max:50'
            ]);

            $searchQuery = $validated['q'] ?? '';
            $categoryId = $validated['category_id'] ?? null;
            $subcategoryId = $validated['subcategory_id'] ?? null;
            $priceMin = $validated['price_min'] ?? null;
            $priceMax = $validated['price_max'] ?? null;
            $level = $validated['level'] ?? null;
            $duration = $validated['duration'] ?? null;
            $priceType = $validated['price_type'] ?? null;
            $minRating = $validated['rating'] ?? null;
            $language = $validated['language'] ?? 'en';
            $sortBy = $validated['sort_by'] ?? 'created_at';
            $sortOrder = $validated['sort_order'] ?? 'desc';
            $page = $validated['page'] ?? 1;
            $limit = $validated['limit'] ?? 12;

            // Build query
            $query = Course::with(['category', 'subcategory'])
                ->where('is_active', true);

            // Search by title and description
            if ($searchQuery) {
                $query->where(function ($q) use ($searchQuery) {
                    $q->where('title', 'like', "%{$searchQuery}%")
                      ->orWhere('title_sw', 'like', "%{$searchQuery}%")
                      ->orWhere('description', 'like', "%{$searchQuery}%")
                      ->orWhere('description_sw', 'like', "%{$searchQuery}%");
                });
            }

            // Filter by category
            if ($categoryId) {
                $query->where('category_id', $categoryId);
            }

            // Filter by subcategory
            if ($subcategoryId) {
                $query->where('subcategory_id', $subcategoryId);
            }

            // Filter by price range
            if ($priceMin !== null) {
                $query->where('price', '>=', $priceMin);
            }
            if ($priceMax !== null) {
                $query->where('price', '<=', $priceMax);
            }

            // Filter by level
            if ($level) {
                $query->where('level', ucfirst($level));
            }

            // Filter by duration  
            if ($duration) {
                switch($duration) {
                    case 'short':
                        $query->where('duration_weeks', '<=', 2);
                        break;
                    case 'medium':
                        $query->whereBetween('duration_weeks', [3, 6]);
                        break;
                    case 'long':
                        $query->where('duration_weeks', '>', 6);
                        break;
                }
            }

            // Filter by price type
            if ($priceType) {
                if ($priceType === 'free') {
                    $query->where('price', 0);
                } else {
                    $query->where('price', '>', 0);
                }
            }

            // Filter by minimum rating
            if ($minRating) {
                $query->where('rating_average', '>=', $minRating);
            }

            // Filter by language
            if ($language && $language !== 'en') {
                $query->where('language', $language);
            }

            // Apply sorting
            $query->orderBy($sortBy, $sortOrder);

            // Get total count before pagination
            $totalCourses = $query->count();

            // Apply pagination and get results
            $courses = $query->skip(($page - 1) * $limit)
                           ->take($limit)
                           ->get()
                           ->map(function ($course) use ($language) {
                               return [
                                   'id' => $course->id,
                                   'title' => $course->getLocalizedTitle($language),
                                   'description' => $course->getLocalizedDescription($language),
                                   'thumbnail_url' => $course->thumbnail_url,
                                   'price' => $course->price,
                                   'original_price' => $course->original_price,
                                   'currency' => $course->currency,
                                   'discount_percentage' => $course->discount_percentage,
                                   'duration' => $course->duration,
                                   'difficulty_level' => $course->difficulty_level,
                                   'language' => $course->language,
                                   'rating' => $course->rating,
                                   'review_count' => $course->review_count,
                                   'enrollment_count' => $course->enrollment_count,
                                   'category' => [
                                       'id' => $course->category->id,
                                       'name' => $course->category->getLocalizedName($language)
                                   ],
                                   'subcategory' => $course->subcategory ? [
                                       'id' => $course->subcategory->id,
                                       'name' => $course->subcategory->getLocalizedName($language)
                                   ] : null,
                                   'is_featured' => $course->is_featured,
                                   'certificate_offered' => $course->certificate_offered,
                                   'has_quizzes' => $course->has_quizzes,
                                   'has_assignments' => $course->has_assignments,
                                   'has_downloadable_resources' => $course->has_downloadable_resources,
                                   'created_at' => $course->created_at->toISOString()
                               ];
                           });

            return response()->json([
                'success' => true,
                'data' => [
                    'courses' => $courses,
                    'search_query' => $searchQuery,
                    'filters' => [
                        'category_id' => $categoryId,
                        'subcategory_id' => $subcategoryId,
                        'price_range' => ['min' => $priceMin, 'max' => $priceMax],
                        'level' => $level,
                        'language' => $language,
                        'sort_by' => $sortBy,
                        'sort_order' => $sortOrder
                    ],
                    'pagination' => [
                        'current_page' => $page,
                        'total_pages' => ceil($totalCourses / $limit),
                        'total_items' => $totalCourses,
                        'items_per_page' => $limit,
                        'has_next' => $page < ceil($totalCourses / $limit),
                        'has_prev' => $page > 1
                    ]
                ],
                'message' => 'Course search completed successfully'
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to search courses',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get all courses with filters
     */
    public function getAllCourses(Request $request): JsonResponse
    {
        try {
            $language = $request->get('language', 'en');
            $page = $request->get('page', 1);
            $limit = $request->get('limit', 12);
            $sortBy = $request->get('sort_by', 'created_at');
            $sortOrder = $request->get('sort_order', 'desc');
            $featured = $request->get('featured', null);

            $query = Course::with(['category', 'subcategory'])
                ->where('is_active', true);

            // Filter featured courses if requested
            if ($featured !== null) {
                $query->where('is_featured', (bool) $featured);
            }

            // Apply sorting
            $query->orderBy($sortBy, $sortOrder);

            // Get total count before pagination
            $totalCourses = $query->count();

            // Apply pagination
            $courses = $query->skip(($page - 1) * $limit)
                           ->take($limit)
                           ->get()
                           ->map(function ($course) use ($language) {
                               return [
                                   'id' => $course->id,
                                   'title' => $course->getLocalizedTitle($language),
                                   'description' => $course->getLocalizedDescription($language),
                                   'thumbnail_url' => $course->thumbnail_url,
                                   'price' => $course->price,
                                   'original_price' => $course->original_price,
                                   'currency' => $course->currency,
                                   'discount_percentage' => $course->discount_percentage,
                                   'duration' => $course->duration,
                                   'difficulty_level' => $course->difficulty_level,
                                   'language' => $course->language,
                                   'rating' => $course->rating,
                                   'review_count' => $course->review_count,
                                   'enrollment_count' => $course->enrollment_count,
                                   'category' => [
                                       'id' => $course->category->id,
                                       'name' => $course->category->getLocalizedName($language)
                                   ],
                                   'subcategory' => $course->subcategory ? [
                                       'id' => $course->subcategory->id,
                                       'name' => $course->subcategory->getLocalizedName($language)
                                   ] : null,
                                   'is_featured' => $course->is_featured,
                                   'certificate_offered' => $course->certificate_offered,
                                   'has_quizzes' => $course->has_quizzes,
                                   'has_assignments' => $course->has_assignments,
                                   'has_downloadable_resources' => $course->has_downloadable_resources,
                                   'created_at' => $course->created_at->toISOString()
                               ];
                           });

            return response()->json([
                'success' => true,
                'data' => [
                    'courses' => $courses,
                    'pagination' => [
                        'current_page' => $page,
                        'total_pages' => ceil($totalCourses / $limit),
                        'total_items' => $totalCourses,
                        'items_per_page' => $limit,
                        'has_next' => $page < ceil($totalCourses / $limit),
                        'has_prev' => $page > 1
                    ]
                ],
                'message' => 'Courses retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch courses',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get course lessons/modules for mobile app
     */
    public function getCourseLessons(Request $request, string $courseId): JsonResponse
    {
        try {
            $language = $request->get('language', 'en');
            
            // Verify course exists
            $course = Course::with(['modules' => function($query) {
                $query->where('is_active', true)->orderBy('order_index');
            }])->find($courseId);

            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $lessons = $course->modules->map(function($module) use ($language) {
                return [
                    'id' => $module->id,
                    'title' => $language === 'sw' && $module->title_sw ? $module->title_sw : $module->title,
                    'titleSw' => $module->title_sw,
                    'description' => $language === 'sw' && $module->description_sw ? $module->description_sw : $module->description,
                    'duration' => $module->duration_minutes,
                    'orderIndex' => $module->order_index,
                    'type' => 'video', // Default type
                    'videoUrl' => $module->video_url,
                    'youtubeUrl' => $module->youtube_url,
                    'isPreview' => $module->is_free_preview,
                    'pdfFilePath' => $module->pdf_file_path,
                    'pdfFileName' => $module->pdf_file_name,
                    'learningObjectives' => $module->learning_objectives ?? [],
                    'resources' => $module->resources ?? []
                ];
            });

            return response()->json([
                'success' => true,
                'data' => [
                    'lessons' => $lessons
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch course lessons',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Submit course review
     */
    public function submitReview(Request $request, string $courseId): JsonResponse
    {
        try {
            $firebaseUid = $request->header('X-Firebase-UID');
            
            if (!$firebaseUid) {
                return response()->json([
                    'success' => false,
                    'message' => 'Firebase UID is required'
                ], 401);
            }

            $validated = $request->validate([
                'rating' => 'required|integer|min:1|max:5',
                'title' => 'nullable|string|max:255',
                'comment' => 'required|string|max:1000',
                'commentSw' => 'nullable|string|max:1000'
            ]);

            $course = Course::find($courseId);
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $user = \App\Models\User::where('firebase_uid', $firebaseUid)->first();
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found. Please sync your account first.'
                ], 404);
            }

            // Check if user has already reviewed this course
            $existingReview = CourseReview::where('user_id', $user->id)
                ->where('course_id', $courseId)
                ->first();

            if ($existingReview) {
                return response()->json([
                    'success' => false,
                    'message' => 'You have already reviewed this course'
                ], 400);
            }

            $review = CourseReview::create([
                'user_id' => $user->id,
                'course_id' => $courseId,
                'rating' => $validated['rating'],
                'title' => $validated['title'],
                'review' => $validated['comment'],
                'review_sw' => $validated['commentSw'],
                'is_approved' => true, // Auto-approve for MVP
                'created_at' => now(),
                'updated_at' => now()
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Review submitted successfully',
                'data' => [
                    'reviewId' => $review->id,
                    'status' => 'published'
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to submit review',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get latest courses
     */
    public function getLatestCourses(Request $request): JsonResponse
    {
        try {
            $language = $request->get('language', 'en');
            $limit = $request->get('limit', 10);

            $courses = Course::with(['category', 'subcategory'])
                ->where('is_active', true)
                ->orderBy('created_at', 'desc')
                ->limit($limit)
                ->get()
                ->map(function ($course) use ($language) {
                    return [
                        'id' => $course->id,
                        'title' => $language === 'sw' && $course->title_sw ? $course->title_sw : $course->title,
                        'description' => $language === 'sw' && $course->description_sw ? $course->description_sw : $course->description,
                        'price' => $course->price,
                        'currency' => $course->currency,
                        'rating' => $course->rating_average,
                        'reviewCount' => $course->review_count,
                        'imageUrl' => $course->image_url,
                        'category' => [
                            'id' => $course->category->id,
                            'name' => $language === 'sw' && $course->category->name_sw ? $course->category->name_sw : $course->category->name
                        ],
                        'createdAt' => $course->created_at->toISOString()
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'courses' => $courses
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch latest courses',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get search suggestions
     */
    public function getSearchSuggestions(Request $request): JsonResponse
    {
        try {
            $query = $request->get('q', '');
            
            if (strlen($query) < 2) {
                return response()->json([
                    'success' => true,
                    'data' => [
                        'suggestions' => []
                    ]
                ]);
            }

            // Get suggestions from course titles and common search terms
            $courseTitles = Course::where('is_active', true)
                ->where(function($q) use ($query) {
                    $q->where('title', 'like', "%{$query}%")
                      ->orWhere('title_sw', 'like', "%{$query}%");
                })
                ->limit(5)
                ->pluck('title')
                ->toArray();

            // Add common search terms
            $commonTerms = [
                'export business',
                'export documentation',
                'international trade',
                'business law',
                'logistics',
                'customs procedures'
            ];

            $filteredTerms = array_filter($commonTerms, function($term) use ($query) {
                return stripos($term, $query) !== false;
            });

            $suggestions = array_unique(array_merge($courseTitles, $filteredTerms));
            $suggestions = array_slice($suggestions, 0, 8);

            return response()->json([
                'success' => true,
                'data' => [
                    'suggestions' => array_values($suggestions)
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch search suggestions',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Legacy method - kept for compatibility
     */
    public function getFeaturedCourses(Request $request): JsonResponse
    {
        try {
            $language = $request->get('language', 'en');
            $limit = $request->get('limit', 6);

            $courses = Course::with(['category', 'subcategory'])
                ->where('is_active', true)
                ->where('is_featured', true)
                ->orderBy('enrollment_count', 'desc')
                ->limit($limit)
                ->get()
                ->map(function ($course) use ($language) {
                    return [
                        'id' => $course->id,
                        'title' => $course->getLocalizedTitle($language),
                        'description' => $course->getLocalizedDescription($language),
                        'price' => $course->price,
                        'original_price' => $course->original_price,
                        'currency' => $course->currency,
                        'discount_percentage' => $course->discount_percentage,
                        'rating' => $course->rating,
                        'review_count' => $course->review_count,
                        'duration' => $course->duration,
                        'difficulty_level' => $course->difficulty_level,
                        'thumbnail_url' => $course->thumbnail_url,
                        'enrollment_count' => $course->enrollment_count,
                        'is_bestseller' => $course->enrollment_count > 100,
                        'is_new' => $course->created_at->isAfter(now()->subDays(30)),
                        'badge_text' => $course->enrollment_count > 100 ? 'Bestseller' : ($course->created_at->isAfter(now()->subDays(30)) ? 'New' : 'Popular'),
                        'category' => [
                            'id' => $course->category->id,
                            'name' => $course->category->getLocalizedName($language)
                        ],
                        'certificate_offered' => $course->certificate_offered,
                        'has_quizzes' => $course->has_quizzes,
                        'has_assignments' => $course->has_assignments,
                        'has_downloadable_resources' => $course->has_downloadable_resources
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'featured_courses' => $courses,
                    'total_featured' => $courses->count(),
                    'language' => $language
                ],
                'message' => 'Featured courses retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch featured courses',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }
}