<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\Validation\ValidationException;

class UserController extends Controller
{
    /**
     * List/search users with filters (active/inactive/banned) and pagination.
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'search' => 'nullable|string|max:255',
                'status' => ['nullable', Rule::in(['active', 'inactive', 'banned'])],
                'role'   => 'nullable|string|max:50',
                'page'   => 'nullable|integer|min:1',
                'limit'  => 'nullable|integer|min:1|max:100',
                'sort'   => ['nullable', Rule::in(['created_at', 'updated_at', 'name', 'email'])],
                'dir'    => ['nullable', Rule::in(['asc', 'desc'])],
            ]);

            $page  = $validated['page']  ?? 1;
            $limit = $validated['limit'] ?? 20;
            $sort  = $validated['sort']  ?? 'created_at';
            $dir   = $validated['dir']   ?? 'desc';

            $q = User::query();

            // Search by name/full_name/email/phone/firebase_uid
            if (!empty($validated['search'])) {
                $term = $validated['search'];
                $q->where(function ($qq) use ($term) {
                    $qq->where('name', 'like', "%{$term}%")
                       ->orWhere('full_name', 'like', "%{$term}%")
                       ->orWhere('email', 'like', "%{$term}%")
                       ->orWhere('phone', 'like', "%{$term}%")
                       ->orWhere('firebase_uid', 'like', "%{$term}%");
                });
            }

            // Status filter
            if (!empty($validated['status'])) {
                switch ($validated['status']) {
                    case 'active':
                        $q->where('is_active', true)->where('is_verified', true);
                        break;
                    case 'inactive':
                        $q->where(function ($qq) {
                            $qq->where('is_active', false)->orWhere('is_verified', false);
                        });
                        break;
                    case 'banned':
                        // You don't have an explicit "banned" column; treat as inactive
                        $q->where('is_active', false);
                        break;
                }
            }

            // Role filter
            if (!empty($validated['role'])) {
                $q->where('role', $validated['role']);
            }

            $total = $q->count();

            $users = $q->orderBy($sort, $dir)
                ->skip(($page - 1) * $limit)
                ->take($limit)
                ->get()
                ->map(fn ($u) => $this->mapUser($u));

            return response()->json([
                'success' => true,
                'data' => [
                    'users' => $users,
                    'pagination' => [
                        'currentPage'  => $page,
                        'totalPages'   => (int) ceil($total / $limit),
                        'totalItems'   => $total,
                        'itemsPerPage' => $limit,
                        'hasNext'      => $page < ceil($total / $limit),
                        'hasPrev'      => $page > 1,
                    ],
                    'summary' => [
                        'totalUsers'     => User::count(),
                        'activeUsers'    => User::where('is_active', true)->count(),
                        'verifiedUsers'  => User::where('is_verified', true)->count(),
                        'instructors'    => User::where('role', 'instructor')->count(),
                    ],
                ],
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors'  => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch users',
                'error'   => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    /**
     * View user details by numeric ID.
     */
    public function show(string $userId): JsonResponse
    {
        try {
            /** @var User $user */
            $user = User::findOrFail($userId);

            // Basic stats (guarded if relations don’t exist)
            $enrolledCount   = method_exists($user, 'enrollments') ? $user->enrollments()->count() : 0;
            $completedCount  = 0; // replace when you store completion state in enrollments
            $totalLearnMins  = 0; // replace when you track learning time

            $payload = $this->mapUser($user);
            $payload['stats'] = [
                'enrolledCourses'   => $enrolledCount,
                'completedCourses'  => $completedCount,
                'totalLearningTime' => $totalLearnMins,
            ];

            return response()->json([
                'success' => true,
                'data'    => ['user' => $payload],
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'User not found',
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch user details',
                'error'   => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    /**
     * Create a new user (admin action).
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'fullName'  => 'required|string|max:255',
                'email'     => ['required','email','max:255', Rule::unique('users','email')],
                'phone'     => 'nullable|string|max:20',
                'language'  => ['nullable', Rule::in(['en','sw'])],
                'role'      => ['required', Rule::in(['student','instructor','admin'])],
                'isActive'  => 'nullable|boolean',
                'firebaseUid' => 'nullable|string|max:255',
                'profileImage' => 'nullable|url',
                'isVerified' => 'nullable|boolean',
                'password'  => 'nullable|string|min:6',
            ]);

            $user = new User();
            $user->name              = $validated['fullName'];
            $user->full_name         = $validated['fullName'];
            $user->email             = $validated['email'];
            $user->password          = Hash::make($validated['password'] ?? Str::random(12));
            $user->role              = $validated['role'];
            $user->phone             = $validated['phone'] ?? null;
            $user->language          = $validated['language'] ?? 'en';
            // enum('email','google') → pick a valid value:
            $user->auth_provider     = 'email';
            $user->firebase_uid      = $validated['firebaseUid'] ?? null;
            $user->profile_image_url = $validated['profileImage'] ?? null;
            $user->is_verified       = (bool) ($validated['isVerified'] ?? false);
            $user->is_active         = (bool) ($validated['isActive'] ?? true);
            $user->last_sync_at      = now();
            $user->save();

            return response()->json([
                'success' => true,
                'message' => 'User created successfully',
                'data'    => ['user' => $this->mapUser($user)],
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors'  => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create user',
                'error'   => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    /**
     * Update user details.
     */
    public function update(Request $request, string $userId): JsonResponse
    {
        try {
            $validated = $request->validate([
                'fullName'    => 'sometimes|string|max:255',
                'email'       => ['sometimes','email','max:255', Rule::unique('users','email')->ignore($userId)],
                'phone'       => 'sometimes|nullable|string|max:20',
                'language'    => ['sometimes', Rule::in(['en','sw'])],
                'role'        => ['sometimes', Rule::in(['student','instructor','admin'])],
                'isVerified'  => 'sometimes|boolean',
                'isActive'    => 'sometimes|boolean',
                'firebaseUid' => 'sometimes|nullable|string|max:255',
                'profileImage'=> 'sometimes|nullable|url',
                'password'    => 'sometimes|nullable|string|min:6',
            ]);

            /** @var User $user */
            $user = User::findOrFail($userId);

            if (array_key_exists('fullName', $validated)) {
                $user->name = $validated['fullName'];
                $user->full_name = $validated['fullName'];
            }
            if (array_key_exists('email', $validated)) {
                $user->email = $validated['email'];
            }
            if (array_key_exists('phone', $validated)) {
                $user->phone = $validated['phone'];
            }
            if (array_key_exists('language', $validated)) {
                $user->language = $validated['language'];
            }
            if (array_key_exists('role', $validated)) {
                $user->role = $validated['role'];
            }
            if (array_key_exists('isVerified', $validated)) {
                $user->is_verified = (bool) $validated['isVerified'];
            }
            if (array_key_exists('isActive', $validated)) {
                $user->is_active = (bool) $validated['isActive'];
            }
            if (array_key_exists('firebaseUid', $validated)) {
                $user->firebase_uid = $validated['firebaseUid'];
            }
            if (array_key_exists('profileImage', $validated)) {
                $user->profile_image_url = $validated['profileImage'];
            }
            if (array_key_exists('password', $validated) && $validated['password']) {
                $user->password = Hash::make($validated['password']);
            }

            $user->last_sync_at = now();
            $user->save();

            return response()->json([
                'success' => true,
                'message' => 'User updated successfully',
                'data'    => ['user' => $this->mapUser($user)],
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors'  => $e->errors(),
            ], 422);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'User not found',
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update user',
                'error'   => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    /**
     * Activate/Deactivate/Ban (ban treated as inactive here).
     */
    public function updateStatus(Request $request, string $userId): JsonResponse
    {
        try {
            $validated = $request->validate([
                'status' => ['required', Rule::in(['active','inactive','banned'])],
                'reason' => 'nullable|string|max:500',
            ]);

            /** @var User $user */
            $user = User::findOrFail($userId);

            switch ($validated['status']) {
                case 'active':
                    $user->is_active = true;
                    $user->is_verified = $user->is_verified ?: true;
                    break;
                case 'inactive':
                case 'banned':
                    $user->is_active = false;
                    break;
            }

            $user->last_sync_at = now();
            $user->save();

            $message = match ($validated['status']) {
                'active'   => 'User activated successfully',
                'inactive' => 'User deactivated successfully',
                'banned'   => 'User banned successfully',
            };

            return response()->json([
                'success' => true,
                'message' => $message,
                'data'    => ['user' => $this->mapUser($user)],
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors'  => $e->errors(),
            ], 422);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'User not found',
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update user status',
                'error'   => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    /**
     * Delete user (hard delete).
     * Consider switching to SoftDeletes if you need recovery later.
     */
    public function destroy(string $userId): JsonResponse
    {
        try {
            $user = User::findOrFail($userId);
            $user->delete();

            return response()->json([
                'success' => true,
                'message' => 'User deleted successfully',
                'data'    => [
                    'deletedUserId' => (string) $userId,
                    'deletedAt'     => now()->toISOString(),
                ],
            ]);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'success' => false,
                'message' => 'User not found',
            ], 404);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete user',
                'error'   => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    /**
     * Get list of instructors with basic stats.
     */
    public function getInstructors(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'search' => 'nullable|string|max:255',
                'status' => ['nullable', Rule::in(['active','inactive'])],
                'page'   => 'nullable|integer|min:1',
                'limit'  => 'nullable|integer|min:1|max:100',
            ]);

            $page  = $validated['page']  ?? 1;
            $limit = $validated['limit'] ?? 50;

            $q = User::query()->where('role', 'instructor');

            if (!empty($validated['search'])) {
                $term = $validated['search'];
                $q->where(function ($qq) use ($term) {
                    $qq->where('full_name', 'like', "%{$term}%")
                       ->orWhere('name', 'like', "%{$term}%")
                       ->orWhere('email', 'like', "%{$term}%");
                });
            }

            if (!empty($validated['status'])) {
                if ($validated['status'] === 'active') {
                    $q->where('is_active', true)->where('is_verified', true);
                } else {
                    $q->where('is_active', false);
                }
            }

            $total = $q->count();

            $instructors = $q->orderBy('created_at', 'desc')
                ->skip(($page - 1) * $limit)
                ->take($limit)
                ->get()
                ->map(function (User $u) {
                    return [
                        'id'            => $u->id,
                        'firebaseUid'   => $u->firebase_uid,
                        'fullName'      => $u->full_name ?? $u->name,
                        'email'         => $u->email,
                        'phone'         => $u->phone,
                        'profileImage'  => $u->profile_image_url ?: "https://ui-avatars.com/api/?name=" . urlencode($u->full_name ?: 'Instructor') . "&size=100&background=7C3AED&color=fff",
                        'language'      => $u->language,
                        'isVerified'    => (bool) $u->is_verified,
                        'isActive'      => (bool) $u->is_active,
                        'lastLoginAt'   => $u->last_login_at?->toISOString(),
                        'createdAt'     => $u->created_at?->toISOString(),
                        'updatedAt'     => $u->updated_at?->toISOString(),
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'instructors' => $instructors,
                    'pagination' => [
                        'currentPage'  => $page,
                        'totalPages'   => (int) ceil($total / $limit),
                        'totalItems'   => $total,
                        'itemsPerPage' => $limit,
                        'hasNext'      => $page < ceil($total / $limit),
                        'hasPrev'      => $page > 1,
                    ],
                ],
                'message' => 'Instructors retrieved successfully',
            ]);
        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors'  => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch instructors',
                'error'   => config('app.debug') ? $e->getMessage() : 'Internal server error',
            ], 500);
        }
    }

    /**
     * Map a User model to the API response schema.
     */
    private function mapUser(User $u): array
    {
        return [
            'id'            => $u->id,
            'firebaseUid'   => $u->firebase_uid,
            'fullName'      => $u->full_name ?? $u->name,
            'email'         => $u->email,
            'phone'         => $u->phone,
            'profileImage'  => $u->profile_image_url,
            'language'      => $u->language,
            'authProvider'  => $u->auth_provider,
            'googleId'      => $u->google_id,
            'isVerified'    => (bool) $u->is_verified,
            'isActive'      => (bool) $u->is_active,
            'role'          => $u->role,
            'createdAt'     => $u->created_at?->toISOString(),
            'updatedAt'     => $u->updated_at?->toISOString(),
            'lastLoginAt'   => $u->last_login_at?->toISOString(),
            'lastSyncAt'    => $u->last_sync_at?->toISOString(),
        ];
    }
}