<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Subcategory;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;

class SubcategoryController extends Controller
{
    /**
     * Display a listing of subcategories.
     */
    public function index(Request $request): JsonResponse
    {
        $query = Subcategory::with('category');

        // Filter by category
        if ($request->has('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        // Filter by status
        if ($request->has('status')) {
            if ($request->status === 'active') {
                $query->where('is_active', true);
            } elseif ($request->status === 'inactive') {
                $query->where('is_active', false);
            }
        }

        // Search functionality
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('name_sw', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhere('description_sw', 'like', "%{$search}%");
            });
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'order_index');
        $sortOrder = $request->get('sort_order', 'asc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->get('per_page', 15);
        $subcategories = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $subcategories,
            'message' => 'Subcategories retrieved successfully'
        ]);
    }

    /**
     * Store a newly created subcategory.
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'category_id' => 'required|string|exists:categories,id',
            'name' => 'required|string|max:255',
            'name_sw' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'description_sw' => 'nullable|string',
            'icon' => 'nullable|string|max:255',
            'color' => 'nullable|string|max:7|regex:/^#[0-9A-Fa-f]{6}$/',
            'order_index' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        // Generate unique ID
        $validated['id'] = Str::uuid();

        // Set default order_index if not provided
        if (!isset($validated['order_index'])) {
            $maxOrder = Subcategory::where('category_id', $validated['category_id'])
                                  ->max('order_index') ?? 0;
            $validated['order_index'] = $maxOrder + 1;
        }

        // Set default values
        $validated['is_active'] = $validated['is_active'] ?? true;
        $validated['course_count'] = 0;

        $subcategory = Subcategory::create($validated);

        return response()->json([
            'success' => true,
            'data' => $subcategory->load('category'),
            'message' => 'Subcategory created successfully'
        ], 201);
    }

    /**
     * Display the specified subcategory.
     */
    public function show(string $id): JsonResponse
    {
        $subcategory = Subcategory::with(['category', 'courses'])->find($id);

        if (!$subcategory) {
            return response()->json([
                'success' => false,
                'message' => 'Subcategory not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $subcategory,
            'message' => 'Subcategory retrieved successfully'
        ]);
    }

    /**
     * Update the specified subcategory.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        $subcategory = Subcategory::find($id);

        if (!$subcategory) {
            return response()->json([
                'success' => false,
                'message' => 'Subcategory not found'
            ], 404);
        }

        $validated = $request->validate([
            'category_id' => 'sometimes|required|string|exists:categories,id',
            'name' => 'sometimes|required|string|max:255',
            'name_sw' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'description_sw' => 'nullable|string',
            'icon' => 'nullable|string|max:255',
            'color' => 'nullable|string|max:7|regex:/^#[0-9A-Fa-f]{6}$/',
            'order_index' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
        ]);

        $subcategory->update($validated);

        return response()->json([
            'success' => true,
            'data' => $subcategory->load('category'),
            'message' => 'Subcategory updated successfully'
        ]);
    }

    /**
     * Remove the specified subcategory.
     */
    public function destroy(string $id): JsonResponse
    {
        $subcategory = Subcategory::find($id);

        if (!$subcategory) {
            return response()->json([
                'success' => false,
                'message' => 'Subcategory not found'
            ], 404);
        }

        // Check if subcategory has courses
        if ($subcategory->courses()->count() > 0) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete subcategory with associated courses'
            ], 422);
        }

        $subcategory->delete();

        return response()->json([
            'success' => true,
            'message' => 'Subcategory deleted successfully'
        ]);
    }

    /**
     * Update subcategory status.
     */
    public function updateStatus(Request $request, string $id): JsonResponse
    {
        $subcategory = Subcategory::find($id);

        if (!$subcategory) {
            return response()->json([
                'success' => false,
                'message' => 'Subcategory not found'
            ], 404);
        }

        $validated = $request->validate([
            'is_active' => 'required|boolean'
        ]);

        $subcategory->update($validated);

        return response()->json([
            'success' => true,
            'data' => $subcategory,
            'message' => 'Subcategory status updated successfully'
        ]);
    }

    /**
     * Reorder subcategories within a category.
     */
    public function reorder(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'category_id' => 'required|string|exists:categories,id',
            'subcategories' => 'required|array',
            'subcategories.*.id' => 'required|string|exists:subcategories,id',
            'subcategories.*.order_index' => 'required|integer|min:0'
        ]);

        foreach ($validated['subcategories'] as $subcategoryData) {
            Subcategory::where('id', $subcategoryData['id'])
                      ->where('category_id', $validated['category_id'])
                      ->update(['order_index' => $subcategoryData['order_index']]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Subcategories reordered successfully'
        ]);
    }

    /**
     * Get subcategories by category.
     */
    public function getByCategory(string $categoryId): JsonResponse
    {
        $category = Category::find($categoryId);

        if (!$category) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found'
            ], 404);
        }

        $subcategories = Subcategory::where('category_id', $categoryId)
                                   ->ordered()
                                   ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'category' => $category,
                'subcategories' => $subcategories
            ],
            'message' => 'Subcategories retrieved successfully'
        ]);
    }

    /**
     * Get subcategory statistics.
     */
    public function statistics(): JsonResponse
    {
        $stats = [
            'total_subcategories' => Subcategory::count(),
            'active_subcategories' => Subcategory::where('is_active', true)->count(),
            'inactive_subcategories' => Subcategory::where('is_active', false)->count(),
            'subcategories_with_courses' => Subcategory::where('course_count', '>', 0)->count(),
            'subcategories_by_category' => Subcategory::selectRaw('category_id, COUNT(*) as count')
                                                     ->groupBy('category_id')
                                                     ->with('category:id,name')
                                                     ->get()
                                                     ->map(function ($item) {
                                                         return [
                                                             'category_name' => $item->category->name,
                                                             'count' => $item->count
                                                         ];
                                                     })
        ];

        return response()->json([
            'success' => true,
            'data' => $stats,
            'message' => 'Subcategory statistics retrieved successfully'
        ]);
    }
}