<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;

class SettingController extends Controller
{
    /**
     * Display a listing of settings.
     */
    public function index(Request $request): JsonResponse
    {
        $query = Setting::query();

        // Filter by group
        if ($request->has('group')) {
            $query->where('group', $request->group);
        }

        // Filter by type
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        // Filter by public status
        if ($request->has('public')) {
            $query->where('is_public', $request->boolean('public'));
        }

        // Search functionality
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('key', 'like', "%{$search}%")
                  ->orWhere('label', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'group');
        $sortOrder = $request->get('sort_order', 'asc');
        $query->orderBy($sortBy, $sortOrder)->orderBy('key', 'asc');

        // Pagination
        $perPage = $request->get('per_page', 50);
        $settings = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $settings,
            'message' => 'Settings retrieved successfully'
        ]);
    }

    /**
     * Store a newly created setting.
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'key' => 'required|string|max:255|unique:settings,key',
            'value' => 'required',
            'type' => 'required|string|in:string,integer,float,boolean,array,json,text',
            'group' => 'required|string|max:100',
            'label' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_public' => 'boolean',
            'validation_rules' => 'nullable|string',
            'options' => 'nullable|array',
        ]);

        // Generate unique ID
        $validated['id'] = Str::uuid();

        // Set default values
        $validated['is_public'] = $validated['is_public'] ?? false;

        // Process value based on type
        $this->processValueByType($validated);

        $setting = Setting::create($validated);

        return response()->json([
            'success' => true,
            'data' => $setting,
            'message' => 'Setting created successfully'
        ], 201);
    }

    /**
     * Display the specified setting.
     */
    public function show(string $id): JsonResponse
    {
        $setting = Setting::find($id);

        if (!$setting) {
            return response()->json([
                'success' => false,
                'message' => 'Setting not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $setting,
            'message' => 'Setting retrieved successfully'
        ]);
    }

    /**
     * Update the specified setting.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        $setting = Setting::find($id);

        if (!$setting) {
            return response()->json([
                'success' => false,
                'message' => 'Setting not found'
            ], 404);
        }

        $validated = $request->validate([
            'key' => 'sometimes|required|string|max:255|unique:settings,key,' . $id,
            'value' => 'sometimes|required',
            'type' => 'sometimes|required|string|in:string,integer,float,boolean,array,json,text',
            'group' => 'sometimes|required|string|max:100',
            'label' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'is_public' => 'boolean',
            'validation_rules' => 'nullable|string',
            'options' => 'nullable|array',
        ]);

        // Process value based on type if both value and type are provided
        if (isset($validated['value']) && isset($validated['type'])) {
            $this->processValueByType($validated);
        } elseif (isset($validated['value'])) {
            // Use existing type
            $validated['type'] = $setting->type;
            $this->processValueByType($validated);
        }

        $setting->update($validated);

        return response()->json([
            'success' => true,
            'data' => $setting,
            'message' => 'Setting updated successfully'
        ]);
    }

    /**
     * Remove the specified setting.
     */
    public function destroy(string $id): JsonResponse
    {
        $setting = Setting::find($id);

        if (!$setting) {
            return response()->json([
                'success' => false,
                'message' => 'Setting not found'
            ], 404);
        }

        $setting->delete();

        return response()->json([
            'success' => true,
            'message' => 'Setting deleted successfully'
        ]);
    }

    /**
     * Get settings by group.
     */
    public function getByGroup(string $group): JsonResponse
    {
        $settings = Setting::where('group', $group)
                          ->orderBy('key')
                          ->get();

        if ($settings->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'No settings found for this group'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $settings,
            'message' => 'Settings retrieved successfully'
        ]);
    }

    /**
     * Update multiple settings at once.
     */
    public function updateBulk(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'settings' => 'required|array',
            'settings.*.id' => 'required|string|exists:settings,id',
            'settings.*.value' => 'required',
        ]);

        $updatedSettings = [];

        foreach ($validated['settings'] as $settingData) {
            $setting = Setting::find($settingData['id']);
            
            if ($setting) {
                $processedData = [
                    'value' => $settingData['value'],
                    'type' => $setting->type
                ];
                
                $this->processValueByType($processedData);
                
                $setting->update(['value' => $processedData['value']]);
                $updatedSettings[] = $setting;
            }
        }

        return response()->json([
            'success' => true,
            'data' => $updatedSettings,
            'message' => 'Settings updated successfully'
        ]);
    }

    /**
     * Get all setting groups.
     */
    public function getGroups(): JsonResponse
    {
        $groups = Setting::select('group')
                        ->distinct()
                        ->orderBy('group')
                        ->pluck('group');

        return response()->json([
            'success' => true,
            'data' => $groups,
            'message' => 'Setting groups retrieved successfully'
        ]);
    }

    /**
     * Get public settings (for frontend).
     */
    public function getPublicSettings(): JsonResponse
    {
        $settings = Setting::where('is_public', true)
                          ->get()
                          ->mapWithKeys(function ($setting) {
                              return [$setting->key => $setting->getTypedValue()];
                          });

        return response()->json([
            'success' => true,
            'data' => $settings,
            'message' => 'Public settings retrieved successfully'
        ]);
    }

    /**
     * Get setting by key.
     */
    public function getByKey(string $key): JsonResponse
    {
        $setting = Setting::where('key', $key)->first();

        if (!$setting) {
            return response()->json([
                'success' => false,
                'message' => 'Setting not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'key' => $setting->key,
                'value' => $setting->getTypedValue(),
                'type' => $setting->type,
                'label' => $setting->label,
                'description' => $setting->description,
            ],
            'message' => 'Setting retrieved successfully'
        ]);
    }

    /**
     * Reset settings to default values.
     */
    public function resetToDefaults(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'group' => 'nullable|string|exists:settings,group',
            'confirm' => 'required|boolean|accepted'
        ]);

        $query = Setting::query();
        
        if (isset($validated['group'])) {
            $query->where('group', $validated['group']);
        }

        // This would reset to predefined default values
        // For now, we'll just return a message
        $affectedCount = $query->count();

        return response()->json([
            'success' => true,
            'message' => "Reset operation would affect {$affectedCount} settings. Default values reset functionality needs to be implemented based on your specific requirements.",
            'affected_count' => $affectedCount
        ]);
    }

    /**
     * Export settings.
     */
    public function export(Request $request): JsonResponse
    {
        $query = Setting::query();

        if ($request->has('group')) {
            $query->where('group', $request->group);
        }

        if ($request->has('public_only') && $request->boolean('public_only')) {
            $query->where('is_public', true);
        }

        $settings = $query->get()->map(function ($setting) {
            return [
                'key' => $setting->key,
                'value' => $setting->getTypedValue(),
                'type' => $setting->type,
                'group' => $setting->group,
                'label' => $setting->label,
                'description' => $setting->description,
                'is_public' => $setting->is_public,
                'validation_rules' => $setting->validation_rules,
                'options' => $setting->options,
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $settings,
            'message' => 'Settings exported successfully'
        ]);
    }

    /**
     * Process value based on type.
     */
    private function processValueByType(array &$data): void
    {
        switch ($data['type']) {
            case 'boolean':
                $data['value'] = (bool) $data['value'];
                break;
            case 'integer':
                $data['value'] = (int) $data['value'];
                break;
            case 'float':
                $data['value'] = (float) $data['value'];
                break;
            case 'array':
            case 'json':
                if (is_string($data['value'])) {
                    $data['value'] = json_decode($data['value'], true);
                }
                break;
            default:
                // string and text types remain as-is
                break;
        }
    }
}