<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;

class ProgressController extends Controller
{
    /**
     * List all user progress/quiz scores
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'course_id' => 'nullable|string|max:100',
                'user_id' => 'nullable|string|max:100',
                'status' => 'nullable|in:not_started,in_progress,completed',
                'min_progress' => 'nullable|integer|min:0|max:100',
                'max_progress' => 'nullable|integer|min:0|max:100',
                'page' => 'nullable|integer|min:1',
                'limit' => 'nullable|integer|min:1|max:100'
            ]);

            // Mock user progress data
            $allProgress = [
                [
                    'id' => 'progress_1',
                    'user' => [
                        'id' => 'user_1',
                        'name' => 'John Mwalimu',
                        'email' => 'john@example.com'
                    ],
                    'course' => [
                        'id' => 'course_export_basics',
                        'title' => 'Export Business Fundamentals',
                        'titleSw' => 'Misingi ya Biashara ya Uuzaji Nje'
                    ],
                    'progress' => 85,
                    'status' => 'in_progress',
                    'enrolledAt' => '2025-01-15T12:00:00Z',
                    'lastAccessedAt' => '2025-01-23T14:00:00Z',
                    'timeSpent' => 1440, // minutes
                    'lessonsCompleted' => 10,
                    'totalLessons' => 12,
                    'quizzesPassed' => 8,
                    'totalQuizzes' => 10,
                    'averageQuizScore' => 88.5,
                    'currentLesson' => [
                        'id' => 'lesson_11',
                        'title' => 'International Shipping Methods',
                        'progress' => 45
                    ]
                ],
                [
                    'id' => 'progress_2',
                    'user' => [
                        'id' => 'user_2',
                        'name' => 'Sarah Kimani',
                        'email' => 'sarah@example.com'
                    ],
                    'course' => [
                        'id' => 'course_digital_marketing',
                        'title' => 'Digital Marketing for Exporters',
                        'titleSw' => 'Uuzaji wa Kidijitali kwa Wauzaji Nje'
                    ],
                    'progress' => 100,
                    'status' => 'completed',
                    'enrolledAt' => '2025-01-18T09:00:00Z',
                    'completedAt' => '2025-01-22T16:30:00Z',
                    'lastAccessedAt' => '2025-01-22T16:30:00Z',
                    'timeSpent' => 2160, // minutes
                    'lessonsCompleted' => 18,
                    'totalLessons' => 18,
                    'quizzesPassed' => 15,
                    'totalQuizzes' => 15,
                    'averageQuizScore' => 92.3,
                    'finalScore' => 87,
                    'certificateEarned' => true
                ],
                [
                    'id' => 'progress_3',
                    'user' => [
                        'id' => 'user_3',
                        'name' => 'David Mbugua',
                        'email' => 'david@example.com'
                    ],
                    'course' => [
                        'id' => 'course_export_basics',
                        'title' => 'Export Business Fundamentals',
                        'titleSw' => 'Misingi ya Biashara ya Uuzaji Nje'
                    ],
                    'progress' => 25,
                    'status' => 'in_progress',
                    'enrolledAt' => '2025-01-20T11:00:00Z',
                    'lastAccessedAt' => '2025-01-21T10:30:00Z',
                    'timeSpent' => 180, // minutes
                    'lessonsCompleted' => 3,
                    'totalLessons' => 12,
                    'quizzesPassed' => 2,
                    'totalQuizzes' => 10,
                    'averageQuizScore' => 75.0,
                    'currentLesson' => [
                        'id' => 'lesson_4',
                        'title' => 'Export Documentation Requirements',
                        'progress' => 0
                    ]
                ]
            ];

            // Apply filters
            $filteredProgress = $allProgress;

            if ($validated['course_id'] ?? null) {
                $filteredProgress = array_filter($filteredProgress, function($progress) use ($validated) {
                    return $progress['course']['id'] === $validated['course_id'];
                });
            }

            if ($validated['user_id'] ?? null) {
                $filteredProgress = array_filter($filteredProgress, function($progress) use ($validated) {
                    return $progress['user']['id'] === $validated['user_id'];
                });
            }

            if ($validated['status'] ?? null) {
                $filteredProgress = array_filter($filteredProgress, function($progress) use ($validated) {
                    return $progress['status'] === $validated['status'];
                });
            }

            // Pagination
            $page = $validated['page'] ?? 1;
            $limit = $validated['limit'] ?? 20;
            $totalProgress = count($filteredProgress);
            $offset = ($page - 1) * $limit;
            $paginatedProgress = array_slice($filteredProgress, $offset, $limit);

            return response()->json([
                'success' => true,
                'data' => [
                    'progress' => array_values($paginatedProgress),
                    'pagination' => [
                        'currentPage' => $page,
                        'totalPages' => ceil($totalProgress / $limit),
                        'totalItems' => $totalProgress,
                        'itemsPerPage' => $limit,
                        'hasNext' => $page < ceil($totalProgress / $limit),
                        'hasPrev' => $page > 1
                    ],
                    'summary' => [
                        'totalEnrollments' => count($allProgress),
                        'inProgress' => count(array_filter($allProgress, fn($p) => $p['status'] === 'in_progress')),
                        'completed' => count(array_filter($allProgress, fn($p) => $p['status'] === 'completed')),
                        'averageProgress' => round(array_sum(array_column($allProgress, 'progress')) / count($allProgress), 1),
                        'averageTimeSpent' => round(array_sum(array_column($allProgress, 'timeSpent')) / count($allProgress), 0)
                    ]
                ]
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch progress data',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get specific user's progress details
     */
    public function show(string $userId): JsonResponse
    {
        try {
            // Mock detailed user progress
            $userProgress = [
                'user' => [
                    'id' => $userId,
                    'name' => 'John Mwalimu',
                    'email' => 'john@example.com',
                    'profileImage' => 'https://ui-avatars.com/api/?name=John+Mwalimu&size=200&background=7C3AED&color=fff',
                    'joinedAt' => '2025-01-15T10:00:00Z'
                ],
                'overallStats' => [
                    'totalEnrollments' => 3,
                    'coursesCompleted' => 1,
                    'coursesInProgress' => 2,
                    'totalLearningTime' => 3600, // minutes
                    'averageProgress' => 70,
                    'certificatesEarned' => 1,
                    'totalQuizzesTaken' => 23,
                    'averageQuizScore' => 86.5
                ],
                'courseProgress' => [
                    [
                        'courseId' => 'course_export_basics',
                        'courseTitle' => 'Export Business Fundamentals',
                        'courseTitleSw' => 'Misingi ya Biashara ya Uuzaji Nje',
                        'progress' => 85,
                        'status' => 'in_progress',
                        'enrolledAt' => '2025-01-15T12:00:00Z',
                        'lastAccessedAt' => '2025-01-23T14:00:00Z',
                        'timeSpent' => 1440,
                        'lessonsCompleted' => 10,
                        'totalLessons' => 12,
                        'quizzesPassed' => 8,
                        'totalQuizzes' => 10,
                        'averageQuizScore' => 88.5,
                        'currentLesson' => 'International Shipping Methods'
                    ],
                    [
                        'courseId' => 'course_digital_marketing',
                        'courseTitle' => 'Digital Marketing for Exporters',
                        'courseTitleSw' => 'Uuzaji wa Kidijitali kwa Wauzaji Nje',
                        'progress' => 100,
                        'status' => 'completed',
                        'enrolledAt' => '2025-01-18T09:00:00Z',
                        'completedAt' => '2025-01-22T16:30:00Z',
                        'timeSpent' => 2160,
                        'lessonsCompleted' => 18,
                        'totalLessons' => 18,
                        'quizzesPassed' => 15,
                        'totalQuizzes' => 15,
                        'averageQuizScore' => 92.3,
                        'finalScore' => 87,
                        'certificateEarned' => true
                    ]
                ],
                'recentActivity' => [
                    [
                        'type' => 'lesson_completed',
                        'description' => 'Completed lesson: Export Documentation',
                        'courseTitle' => 'Export Business Fundamentals',
                        'timestamp' => '2025-01-23T14:00:00Z'
                    ],
                    [
                        'type' => 'quiz_passed',
                        'description' => 'Passed quiz with 90% score',
                        'courseTitle' => 'Export Business Fundamentals',
                        'score' => 90,
                        'timestamp' => '2025-01-23T13:45:00Z'
                    ],
                    [
                        'type' => 'course_completed',
                        'description' => 'Completed Digital Marketing for Exporters',
                        'courseTitle' => 'Digital Marketing for Exporters',
                        'finalScore' => 87,
                        'timestamp' => '2025-01-22T16:30:00Z'
                    ]
                ],
                'learningStreak' => [
                    'currentStreak' => 7, // days
                    'longestStreak' => 12,
                    'streakStartDate' => '2025-01-17T00:00:00Z'
                ]
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'userProgress' => $userProgress
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch user progress details',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }
}