<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\News;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;

class NewsController extends Controller
{
    /**
     * Display a listing of news articles.
     */
    public function index(Request $request): JsonResponse
    {
        $query = News::query();

        // Filter by publication status
        if ($request->has('status')) {
            if ($request->status === 'published') {
                $query->where('is_published', true);
            } elseif ($request->status === 'draft') {
                $query->where('is_published', false);
            }
        }

        // Filter by featured status
        if ($request->has('featured')) {
            $query->where('is_featured', $request->boolean('featured'));
        }

        // Filter by category
        if ($request->has('category')) {
            $query->where('category', $request->category);
        }

        // Filter by author
        if ($request->has('author')) {
            $query->where('author', 'like', "%{$request->author}%");
        }

        // Filter by date range
        if ($request->has('date_from')) {
            $query->where('published_at', '>=', $request->date_from);
        }
        if ($request->has('date_to')) {
            $query->where('published_at', '<=', $request->date_to);
        }

        // Search functionality
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('title_sw', 'like', "%{$search}%")
                  ->orWhere('content', 'like', "%{$search}%")
                  ->orWhere('content_sw', 'like', "%{$search}%")
                  ->orWhere('excerpt', 'like', "%{$search}%")
                  ->orWhere('excerpt_sw', 'like', "%{$search}%")
                  ->orWhere('author', 'like', "%{$search}%");
            });
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->get('per_page', 15);
        $news = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $news,
            'message' => 'News articles retrieved successfully'
        ]);
    }

    /**
     * Store a newly created news article.
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'title_sw' => 'nullable|string|max:255',
            'content' => 'required|string',
            'content_sw' => 'nullable|string',
            'excerpt' => 'nullable|string|max:500',
            'excerpt_sw' => 'nullable|string|max:500',
            'featured_image' => 'nullable|string|max:500',
            'category' => 'required|string|max:100',
            'author' => 'required|string|max:255',
            'is_featured' => 'boolean',
            'is_published' => 'boolean',
            'published_at' => 'nullable|date',
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'slug' => 'nullable|string|max:255|unique:news,slug',
        ]);

        // Generate unique ID
        $validated['id'] = Str::uuid();

        // Set default values
        $validated['is_featured'] = $validated['is_featured'] ?? false;
        $validated['is_published'] = $validated['is_published'] ?? false;
        $validated['view_count'] = 0;

        // Set published_at if publishing
        if ($validated['is_published'] && !isset($validated['published_at'])) {
            $validated['published_at'] = now();
        }

        // Generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['title']);
            
            // Ensure slug is unique
            $originalSlug = $validated['slug'];
            $counter = 1;
            while (News::where('slug', $validated['slug'])->exists()) {
                $validated['slug'] = $originalSlug . '-' . $counter;
                $counter++;
            }
        }

        $news = News::create($validated);

        return response()->json([
            'success' => true,
            'data' => $news,
            'message' => 'News article created successfully'
        ], 201);
    }

    /**
     * Display the specified news article.
     */
    public function show(string $id): JsonResponse
    {
        $news = News::find($id);

        if (!$news) {
            return response()->json([
                'success' => false,
                'message' => 'News article not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $news,
            'message' => 'News article retrieved successfully'
        ]);
    }

    /**
     * Update the specified news article.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        $news = News::find($id);

        if (!$news) {
            return response()->json([
                'success' => false,
                'message' => 'News article not found'
            ], 404);
        }

        $validated = $request->validate([
            'title' => 'sometimes|required|string|max:255',
            'title_sw' => 'nullable|string|max:255',
            'content' => 'sometimes|required|string',
            'content_sw' => 'nullable|string',
            'excerpt' => 'nullable|string|max:500',
            'excerpt_sw' => 'nullable|string|max:500',
            'featured_image' => 'nullable|string|max:500',
            'category' => 'sometimes|required|string|max:100',
            'author' => 'sometimes|required|string|max:255',
            'is_featured' => 'boolean',
            'is_published' => 'boolean',
            'published_at' => 'nullable|date',
            'tags' => 'nullable|array',
            'tags.*' => 'string|max:50',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'slug' => 'nullable|string|max:255|unique:news,slug,' . $id,
        ]);

        // Set published_at if publishing for the first time
        if (isset($validated['is_published']) && $validated['is_published'] && !$news->is_published) {
            $validated['published_at'] = $validated['published_at'] ?? now();
        }

        // Update slug if title changed
        if (isset($validated['title']) && $validated['title'] !== $news->title && empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['title']);
            
            // Ensure slug is unique
            $originalSlug = $validated['slug'];
            $counter = 1;
            while (News::where('slug', $validated['slug'])->where('id', '!=', $id)->exists()) {
                $validated['slug'] = $originalSlug . '-' . $counter;
                $counter++;
            }
        }

        $news->update($validated);

        return response()->json([
            'success' => true,
            'data' => $news,
            'message' => 'News article updated successfully'
        ]);
    }

    /**
     * Remove the specified news article.
     */
    public function destroy(string $id): JsonResponse
    {
        $news = News::find($id);

        if (!$news) {
            return response()->json([
                'success' => false,
                'message' => 'News article not found'
            ], 404);
        }

        $news->delete();

        return response()->json([
            'success' => true,
            'message' => 'News article deleted successfully'
        ]);
    }

    /**
     * Update news article publication status.
     */
    public function updateStatus(Request $request, string $id): JsonResponse
    {
        $news = News::find($id);

        if (!$news) {
            return response()->json([
                'success' => false,
                'message' => 'News article not found'
            ], 404);
        }

        $validated = $request->validate([
            'is_published' => 'required|boolean',
            'published_at' => 'nullable|date'
        ]);

        // Set published_at if publishing
        if ($validated['is_published'] && !$news->is_published) {
            $validated['published_at'] = $validated['published_at'] ?? now();
        }

        $news->update($validated);

        return response()->json([
            'success' => true,
            'data' => $news,
            'message' => 'News article status updated successfully'
        ]);
    }

    /**
     * Update news article featured status.
     */
    public function updateFeatured(Request $request, string $id): JsonResponse
    {
        $news = News::find($id);

        if (!$news) {
            return response()->json([
                'success' => false,
                'message' => 'News article not found'
            ], 404);
        }

        $validated = $request->validate([
            'is_featured' => 'required|boolean'
        ]);

        $news->update($validated);

        return response()->json([
            'success' => true,
            'data' => $news,
            'message' => 'News article featured status updated successfully'
        ]);
    }

    /**
     * Get news categories.
     */
    public function getCategories(): JsonResponse
    {
        $categories = News::select('category')
                         ->distinct()
                         ->whereNotNull('category')
                         ->orderBy('category')
                         ->pluck('category');

        return response()->json([
            'success' => true,
            'data' => $categories,
            'message' => 'News categories retrieved successfully'
        ]);
    }

    /**
     * Get news statistics.
     */
    public function statistics(): JsonResponse
    {
        $stats = [
            'total_articles' => News::count(),
            'published_articles' => News::where('is_published', true)->count(),
            'draft_articles' => News::where('is_published', false)->count(),
            'featured_articles' => News::where('is_featured', true)->count(),
            'total_views' => News::sum('view_count'),
            'articles_by_category' => News::selectRaw('category, COUNT(*) as count')
                                         ->groupBy('category')
                                         ->orderBy('count', 'desc')
                                         ->get()
                                         ->pluck('count', 'category')
                                         ->toArray(),
            'articles_by_author' => News::selectRaw('author, COUNT(*) as count')
                                       ->groupBy('author')
                                       ->orderBy('count', 'desc')
                                       ->limit(10)
                                       ->get()
                                       ->pluck('count', 'author')
                                       ->toArray(),
            'recent_articles' => News::orderBy('created_at', 'desc')
                                    ->limit(5)
                                    ->get(['id', 'title', 'author', 'is_published', 'created_at'])
        ];

        return response()->json([
            'success' => true,
            'data' => $stats,
            'message' => 'News statistics retrieved successfully'
        ]);
    }

    /**
     * Duplicate a news article.
     */
    public function duplicate(string $id): JsonResponse
    {
        $news = News::find($id);

        if (!$news) {
            return response()->json([
                'success' => false,
                'message' => 'News article not found'
            ], 404);
        }

        $newNews = $news->replicate();
        $newNews->id = Str::uuid();
        $newNews->title = $news->title . ' (Copy)';
        $newNews->slug = Str::slug($newNews->title);
        $newNews->is_published = false;
        $newNews->is_featured = false;
        $newNews->view_count = 0;
        $newNews->published_at = null;
        
        // Ensure slug is unique
        $originalSlug = $newNews->slug;
        $counter = 1;
        while (News::where('slug', $newNews->slug)->exists()) {
            $newNews->slug = $originalSlug . '-' . $counter;
            $counter++;
        }
        
        $newNews->save();

        return response()->json([
            'success' => true,
            'data' => $newNews,
            'message' => 'News article duplicated successfully'
        ], 201);
    }
}