<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;

class MessageController extends Controller
{
    /**
     * List platform/admin messages
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'type' => 'nullable|in:broadcast,individual,system,announcement',
                'status' => 'nullable|in:draft,sent,scheduled',
                'recipient' => 'nullable|string|max:100',
                'page' => 'nullable|integer|min:1',
                'limit' => 'nullable|integer|min:1|max:100'
            ]);

            // Mock admin messages data
            $allMessages = [
                [
                    'id' => 'msg_1',
                    'type' => 'broadcast',
                    'subject' => 'New Course Available: Digital Marketing',
                    'subjectSw' => 'Kozi Mpya Inapatikana: Uuzaji wa Kidijitali',
                    'content' => 'We are excited to announce a new course on Digital Marketing for Exporters...',
                    'contentSw' => 'Tunafurahi kutangaza kozi mpya ya Uuzaji wa Kidijitali kwa Wauzaji Nje...',
                    'recipients' => [
                        'type' => 'all_users',
                        'count' => 1247
                    ],
                    'status' => 'sent',
                    'priority' => 'normal',
                    'sentAt' => '2025-01-23T10:00:00Z',
                    'sentBy' => [
                        'id' => 'admin_1',
                        'name' => 'System Administrator'
                    ],
                    'stats' => [
                        'delivered' => 1247,
                        'opened' => 892,
                        'clicked' => 234,
                        'openRate' => 71.5,
                        'clickRate' => 18.8
                    ],
                    'createdAt' => '2025-01-23T09:30:00Z'
                ],
                [
                    'id' => 'msg_2',
                    'type' => 'individual',
                    'subject' => 'Course Completion Congratulations',
                    'subjectSw' => 'Hongera kwa Kukamilisha Kozi',
                    'content' => 'Congratulations on completing the Export Business Fundamentals course...',
                    'contentSw' => 'Hongera kwa kukamilisha kozi ya Misingi ya Biashara ya Uuzaji Nje...',
                    'recipients' => [
                        'type' => 'individual',
                        'user' => [
                            'id' => 'user_1',
                            'name' => 'John Mwalimu',
                            'email' => 'john@example.com'
                        ]
                    ],
                    'status' => 'sent',
                    'priority' => 'normal',
                    'sentAt' => '2025-01-22T16:45:00Z',
                    'sentBy' => [
                        'id' => 'admin_1',
                        'name' => 'System Administrator'
                    ],
                    'stats' => [
                        'delivered' => 1,
                        'opened' => 1,
                        'clicked' => 1,
                        'openRate' => 100,
                        'clickRate' => 100
                    ],
                    'createdAt' => '2025-01-22T16:30:00Z'
                ],
                [
                    'id' => 'msg_3',
                    'type' => 'announcement',
                    'subject' => 'Platform Maintenance Scheduled',
                    'subjectSw' => 'Marekebisho ya Jukwaa Yamepangwa',
                    'content' => 'We will be performing scheduled maintenance on our platform...',
                    'contentSw' => 'Tutafanya marekebisho yaliyopangwa kwenye jukwaa letu...',
                    'recipients' => [
                        'type' => 'active_users',
                        'count' => 892
                    ],
                    'status' => 'scheduled',
                    'priority' => 'high',
                    'scheduledFor' => '2025-01-25T02:00:00Z',
                    'sentBy' => [
                        'id' => 'admin_1',
                        'name' => 'System Administrator'
                    ],
                    'createdAt' => '2025-01-23T15:00:00Z'
                ],
                [
                    'id' => 'msg_4',
                    'type' => 'system',
                    'subject' => 'Welcome to Exporters Point',
                    'subjectSw' => 'Karibu Exporters Point',
                    'content' => 'Welcome to Exporters Point! We are excited to have you join our community...',
                    'contentSw' => 'Karibu Exporters Point! Tunafurahi kuwa na wewe katika jamii yetu...',
                    'recipients' => [
                        'type' => 'new_users',
                        'trigger' => 'auto_on_registration'
                    ],
                    'status' => 'sent',
                    'priority' => 'normal',
                    'isAutomatic' => true,
                    'stats' => [
                        'delivered' => 45,
                        'opened' => 38,
                        'clicked' => 12,
                        'openRate' => 84.4,
                        'clickRate' => 26.7
                    ],
                    'createdAt' => '2025-01-15T00:00:00Z'
                ]
            ];

            // Apply filters
            $filteredMessages = $allMessages;

            if ($validated['type'] ?? null) {
                $filteredMessages = array_filter($filteredMessages, function($message) use ($validated) {
                    return $message['type'] === $validated['type'];
                });
            }

            if ($validated['status'] ?? null) {
                $filteredMessages = array_filter($filteredMessages, function($message) use ($validated) {
                    return $message['status'] === $validated['status'];
                });
            }

            // Pagination
            $page = $validated['page'] ?? 1;
            $limit = $validated['limit'] ?? 20;
            $totalMessages = count($filteredMessages);
            $offset = ($page - 1) * $limit;
            $paginatedMessages = array_slice($filteredMessages, $offset, $limit);

            return response()->json([
                'success' => true,
                'data' => [
                    'messages' => array_values($paginatedMessages),
                    'pagination' => [
                        'currentPage' => $page,
                        'totalPages' => ceil($totalMessages / $limit),
                        'totalItems' => $totalMessages,
                        'itemsPerPage' => $limit,
                        'hasNext' => $page < ceil($totalMessages / $limit),
                        'hasPrev' => $page > 1
                    ],
                    'summary' => [
                        'totalMessages' => count($allMessages),
                        'sentMessages' => count(array_filter($allMessages, fn($m) => $m['status'] === 'sent')),
                        'scheduledMessages' => count(array_filter($allMessages, fn($m) => $m['status'] === 'scheduled')),
                        'draftMessages' => count(array_filter($allMessages, fn($m) => $m['status'] === 'draft'))
                    ]
                ]
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch messages',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Send message to users
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'type' => 'required|in:broadcast,individual,announcement,system',
                'subject' => 'required|string|max:255',
                'subjectSw' => 'required|string|max:255',
                'content' => 'required|string|max:5000',
                'contentSw' => 'required|string|max:5000',
                'recipients' => 'required|array',
                'recipients.type' => 'required|in:all_users,active_users,inactive_users,instructors,individual,course_enrolled',
                'recipients.user_ids' => 'nullable|array',
                'recipients.course_id' => 'nullable|string|max:100',
                'priority' => 'nullable|in:low,normal,high,urgent',
                'scheduledFor' => 'nullable|date|after:now',
                'sendImmediately' => 'nullable|boolean'
            ]);

            $messageId = 'msg_' . uniqid();
            $sendImmediately = $validated['sendImmediately'] ?? true;
            $scheduledFor = $validated['scheduledFor'] ?? null;

            // Mock message sending
            $newMessage = [
                'id' => $messageId,
                'type' => $validated['type'],
                'subject' => $validated['subject'],
                'subjectSw' => $validated['subjectSw'],
                'content' => $validated['content'],
                'contentSw' => $validated['contentSw'],
                'recipients' => $validated['recipients'],
                'priority' => $validated['priority'] ?? 'normal',
                'status' => $sendImmediately ? 'sent' : ($scheduledFor ? 'scheduled' : 'draft'),
                'sentAt' => $sendImmediately ? now()->toISOString() : null,
                'scheduledFor' => $scheduledFor,
                'sentBy' => [
                    'id' => 'admin_1',
                    'name' => 'Current Admin'
                ],
                'createdAt' => now()->toISOString()
            ];

            // Calculate recipient count based on type
            $recipientCount = match($validated['recipients']['type']) {
                'all_users' => 1247,
                'active_users' => 892,
                'inactive_users' => 355,
                'instructors' => 12,
                'individual' => count($validated['recipients']['user_ids'] ?? []),
                'course_enrolled' => 342, // Mock count for course enrolled users
                default => 0
            };

            $newMessage['recipientCount'] = $recipientCount;

            if ($sendImmediately) {
                $newMessage['stats'] = [
                    'delivered' => $recipientCount,
                    'opened' => 0,
                    'clicked' => 0,
                    'openRate' => 0,
                    'clickRate' => 0
                ];
            }

            $message = $sendImmediately ? 'Message sent successfully' : 
                      ($scheduledFor ? 'Message scheduled successfully' : 'Message saved as draft');

            return response()->json([
                'success' => true,
                'message' => $message,
                'data' => [
                    'message' => $newMessage
                ]
            ], 201);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to send message',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }
}