<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Media;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\UploadedFile;

class MediaController extends Controller
{
    /**
     * Display a listing of media files.
     */
    public function index(Request $request): JsonResponse
    {
        $query = Media::query();

        // Filter by folder
        if ($request->has('folder')) {
            $query->where('folder', $request->folder);
        }

        // Filter by mime type
        if ($request->has('type')) {
            switch ($request->type) {
                case 'images':
                    $query->where('mime_type', 'like', 'image/%');
                    break;
                case 'videos':
                    $query->where('mime_type', 'like', 'video/%');
                    break;
                case 'documents':
                    $query->whereIn('mime_type', [
                        'application/pdf',
                        'application/msword',
                        'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                        'application/vnd.ms-excel',
                        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                        'text/plain',
                    ]);
                    break;
            }
        }

        // Filter by public status
        if ($request->has('public')) {
            $query->where('is_public', $request->boolean('public'));
        }

        // Search functionality
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('filename', 'like', "%{$search}%")
                  ->orWhere('original_filename', 'like', "%{$search}%")
                  ->orWhere('title', 'like', "%{$search}%")
                  ->orWhere('alt_text', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->get('per_page', 20);
        $media = $query->with('uploader:id,name,email')->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $media,
            'message' => 'Media files retrieved successfully'
        ]);
    }

    /**
     * Upload a new media file.
     */
    public function upload(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'file' => 'required|file|max:51200', // 50MB max
            'folder' => 'nullable|string|max:100',
            'title' => 'nullable|string|max:255',
            'alt_text' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'is_public' => 'boolean',
        ]);

        $file = $request->file('file');
        
        // Validate file type
        $allowedMimes = [
            'image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/svg+xml',
            'video/mp4', 'video/avi', 'video/mov', 'video/wmv', 'video/webm',
            'application/pdf', 'application/msword', 
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'text/plain'
        ];

        if (!in_array($file->getMimeType(), $allowedMimes)) {
            return response()->json([
                'success' => false,
                'message' => 'File type not allowed'
            ], 422);
        }

        try {
            $folder = $validated['folder'] ?? 'uploads';
            $filename = $this->generateUniqueFilename($file, $folder);
            
            // Store file
            $path = $file->storeAs("public/media/{$folder}", $filename);
            $url = Storage::url($path);

            // Get file dimensions for images
            $width = null;
            $height = null;
            if (str_starts_with($file->getMimeType(), 'image/')) {
                $imageInfo = getimagesize($file->getPathname());
                if ($imageInfo) {
                    $width = $imageInfo[0];
                    $height = $imageInfo[1];
                }
            }

            // Create media record
            $media = Media::create([
                'id' => Str::uuid(),
                'filename' => $filename,
                'original_filename' => $file->getClientOriginalName(),
                'path' => str_replace('public/', '', $path),
                'url' => $url,
                'mime_type' => $file->getMimeType(),
                'size' => $file->getSize(),
                'width' => $width,
                'height' => $height,
                'title' => $validated['title'] ?? pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME),
                'alt_text' => $validated['alt_text'],
                'description' => $validated['description'],
                'folder' => $folder,
                'is_public' => $validated['is_public'] ?? true,
                'uploaded_by' => auth()->id(),
                'metadata' => [
                    'original_name' => $file->getClientOriginalName(),
                    'extension' => $file->getClientOriginalExtension(),
                    'upload_ip' => $request->ip(),
                    'upload_user_agent' => $request->userAgent(),
                ]
            ]);

            return response()->json([
                'success' => true,
                'data' => $media,
                'message' => 'File uploaded successfully'
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'File upload failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Upload multiple files.
     */
    public function uploadMultiple(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'files' => 'required|array|max:10',
            'files.*' => 'file|max:51200',
            'folder' => 'nullable|string|max:100',
            'is_public' => 'boolean',
        ]);

        $uploadedFiles = [];
        $errors = [];

        foreach ($validated['files'] as $index => $file) {
            try {
                $singleRequest = new Request();
                $singleRequest->files->set('file', $file);
                $singleRequest->merge([
                    'folder' => $validated['folder'] ?? 'uploads',
                    'is_public' => $validated['is_public'] ?? true,
                ]);

                $response = $this->upload($singleRequest);
                $responseData = json_decode($response->getContent(), true);

                if ($responseData['success']) {
                    $uploadedFiles[] = $responseData['data'];
                } else {
                    $errors[] = [
                        'file_index' => $index,
                        'filename' => $file->getClientOriginalName(),
                        'error' => $responseData['message']
                    ];
                }
            } catch (\Exception $e) {
                $errors[] = [
                    'file_index' => $index,
                    'filename' => $file->getClientOriginalName(),
                    'error' => $e->getMessage()
                ];
            }
        }

        return response()->json([
            'success' => count($uploadedFiles) > 0,
            'data' => [
                'uploaded_files' => $uploadedFiles,
                'errors' => $errors,
                'total_uploaded' => count($uploadedFiles),
                'total_errors' => count($errors)
            ],
            'message' => count($uploadedFiles) > 0 
                ? 'Files uploaded successfully' 
                : 'No files were uploaded successfully'
        ], count($uploadedFiles) > 0 ? 201 : 422);
    }

    /**
     * Display the specified media file.
     */
    public function show(string $id): JsonResponse
    {
        $media = Media::with('uploader:id,name,email')->find($id);

        if (!$media) {
            return response()->json([
                'success' => false,
                'message' => 'Media file not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $media,
            'message' => 'Media file retrieved successfully'
        ]);
    }

    /**
     * Update the specified media file metadata.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        $media = Media::find($id);

        if (!$media) {
            return response()->json([
                'success' => false,
                'message' => 'Media file not found'
            ], 404);
        }

        $validated = $request->validate([
            'title' => 'nullable|string|max:255',
            'alt_text' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'folder' => 'nullable|string|max:100',
            'is_public' => 'boolean',
        ]);

        $media->update($validated);

        return response()->json([
            'success' => true,
            'data' => $media,
            'message' => 'Media file updated successfully'
        ]);
    }

    /**
     * Remove the specified media file.
     */
    public function destroy(string $id): JsonResponse
    {
        $media = Media::find($id);

        if (!$media) {
            return response()->json([
                'success' => false,
                'message' => 'Media file not found'
            ], 404);
        }

        try {
            // Delete the actual file
            if (Storage::exists('public/' . $media->path)) {
                Storage::delete('public/' . $media->path);
            }

            // Delete the database record
            $media->delete();

            return response()->json([
                'success' => true,
                'message' => 'Media file deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete media file: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete multiple media files.
     */
    public function destroyMultiple(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'string|exists:media,id'
        ]);

        $deletedCount = 0;
        $errors = [];

        foreach ($validated['ids'] as $id) {
            try {
                $media = Media::find($id);
                if ($media) {
                    // Delete the actual file
                    if (Storage::exists('public/' . $media->path)) {
                        Storage::delete('public/' . $media->path);
                    }
                    $media->delete();
                    $deletedCount++;
                }
            } catch (\Exception $e) {
                $errors[] = [
                    'id' => $id,
                    'error' => $e->getMessage()
                ];
            }
        }

        return response()->json([
            'success' => $deletedCount > 0,
            'data' => [
                'deleted_count' => $deletedCount,
                'errors' => $errors
            ],
            'message' => $deletedCount > 0 
                ? "{$deletedCount} media files deleted successfully" 
                : 'No media files were deleted'
        ]);
    }

    /**
     * Get media folders.
     */
    public function getFolders(): JsonResponse
    {
        $folders = Media::select('folder')
                       ->distinct()
                       ->whereNotNull('folder')
                       ->orderBy('folder')
                       ->pluck('folder');

        return response()->json([
            'success' => true,
            'data' => $folders,
            'message' => 'Media folders retrieved successfully'
        ]);
    }

    /**
     * Get media statistics.
     */
    public function statistics(): JsonResponse
    {
        $stats = [
            'total_files' => Media::count(),
            'total_size' => Media::sum('size'),
            'total_size_formatted' => $this->formatBytes(Media::sum('size')),
            'files_by_type' => [
                'images' => Media::where('mime_type', 'like', 'image/%')->count(),
                'videos' => Media::where('mime_type', 'like', 'video/%')->count(),
                'documents' => Media::documents()->count(),
            ],
            'files_by_folder' => Media::selectRaw('folder, COUNT(*) as count, SUM(size) as total_size')
                                    ->groupBy('folder')
                                    ->orderBy('count', 'desc')
                                    ->get()
                                    ->map(function ($item) {
                                        return [
                                            'folder' => $item->folder,
                                            'count' => $item->count,
                                            'total_size' => $this->formatBytes($item->total_size)
                                        ];
                                    }),
            'recent_uploads' => Media::orderBy('created_at', 'desc')
                                   ->limit(10)
                                   ->get(['id', 'filename', 'mime_type', 'size', 'created_at'])
        ];

        return response()->json([
            'success' => true,
            'data' => $stats,
            'message' => 'Media statistics retrieved successfully'
        ]);
    }

    /**
     * Generate a unique filename.
     */
    private function generateUniqueFilename(UploadedFile $file, string $folder): string
    {
        $extension = $file->getClientOriginalExtension();
        $basename = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        $basename = Str::slug($basename);
        
        $filename = $basename . '.' . $extension;
        $counter = 1;
        
        while (Storage::exists("public/media/{$folder}/{$filename}")) {
            $filename = $basename . '-' . $counter . '.' . $extension;
            $counter++;
        }
        
        return $filename;
    }

    /**
     * Format bytes to human readable format.
     */
    private function formatBytes(int $bytes): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }
}