<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Category;
use App\Models\Subcategory;
use App\Models\Course;
use App\Models\CourseModule;
use App\Models\Banner;
use App\Models\News;
use App\Models\Setting;
use App\Models\Media;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    /**
     * Get admin dashboard overview with comprehensive statistics
     */
    public function overview(Request $request): JsonResponse
    {
        try {
            // Get date range for analytics (default to last 30 days)
            $period = $request->get('period', '30_days');
            $dateRange = $this->getDateRange($period);

            // Core Statistics
            $overview = [
                'users' => $this->getUserStatistics($dateRange),
                'content' => $this->getContentStatistics($dateRange),
                'engagement' => $this->getEngagementStatistics($dateRange),
                'system' => $this->getSystemStatistics(),
                'recent_activity' => $this->getRecentActivity(),
                'charts' => $this->getChartData($period),
                'quick_actions' => $this->getQuickActions(),
                'alerts' => $this->getSystemAlerts()
            ];

            return response()->json([
                'success' => true,
                'data' => $overview,
                'meta' => [
                    'period' => $period,
                    'date_range' => $dateRange,
                    'generated_at' => now()->toISOString(),
                    'cache_duration' => '5_minutes'
                ],
                'message' => 'Dashboard overview retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve dashboard overview',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get user statistics
     */
    private function getUserStatistics(array $dateRange): array
    {
        $totalUsers = User::count();
        $activeUsers = User::where('role', '!=', 'inactive')->count(); // Using role instead of is_active
        $verifiedUsers = User::whereNotNull('email_verified_at')->count();
        $newUsers = User::whereBetween('created_at', $dateRange)->count();
        
        // Role distribution (since we don't have language column)
        $roleStats = User::selectRaw('role, COUNT(*) as count')
                        ->groupBy('role')
                        ->get()
                        ->pluck('count', 'role')
                        ->toArray();

        // Skip auth provider distribution (column doesn't exist)
        $authProviderStats = ['local' => $totalUsers];

        // Recent registrations (last 7 days)
        $recentRegistrations = User::selectRaw('DATE(created_at) as date, COUNT(*) as count')
                                  ->where('created_at', '>=', now()->subDays(7))
                                  ->groupBy('date')
                                  ->orderBy('date')
                                  ->get()
                                  ->pluck('count', 'date')
                                  ->toArray();

        return [
            'total' => $totalUsers,
            'active' => $activeUsers,
            'verified' => $verifiedUsers,
            'new_this_period' => $newUsers,
            'inactive' => $totalUsers - $activeUsers,
            'verification_rate' => $totalUsers > 0 ? round(($verifiedUsers / $totalUsers) * 100, 1) : 0,
            'growth_rate' => $this->calculateGrowthRate('users', $dateRange),
            'role_distribution' => $roleStats,
            'auth_provider_distribution' => $authProviderStats,
            'recent_registrations' => $recentRegistrations
        ];
    }

    /**
     * Get content statistics
     */
    private function getContentStatistics(array $dateRange): array
    {
        // Categories and Subcategories
        $totalCategories = Category::count();
        $activeCategories = Category::where('is_active', true)->count();
        $totalSubcategories = Subcategory::count();
        $activeSubcategories = Subcategory::where('is_active', true)->count();

        // Courses and Modules
        $totalCourses = Course::count();
        $activeCourses = Course::where('is_active', true)->count();
        $featuredCourses = Course::where('is_featured', true)->count();
        $newCourses = Course::whereBetween('created_at', $dateRange)->count();
        
        $totalModules = CourseModule::count();
        $activeModules = CourseModule::where('is_active', true)->count();
        $newModules = CourseModule::whereBetween('created_at', $dateRange)->count();

        // News Articles
        $totalNews = News::count();
        $publishedNews = News::where('is_published', true)->count();
        $draftNews = News::where('is_published', false)->count();
        $featuredNews = News::where('is_featured', true)->count();
        $newArticles = News::whereBetween('created_at', $dateRange)->count();

        // Banners
        $totalBanners = Banner::count();
        $activeBanners = Banner::where('is_active', true)->count();
        $newBanners = Banner::whereBetween('created_at', $dateRange)->count();

        // Media Files
        $totalMedia = Media::count();
        $publicMedia = Media::where('is_public', true)->count();
        $totalMediaSize = Media::sum('size');
        $newMedia = Media::whereBetween('created_at', $dateRange)->count();

        // Media by type
        $mediaByType = Media::selectRaw('
                CASE 
                    WHEN mime_type LIKE "image/%" THEN "images"
                    WHEN mime_type LIKE "video/%" THEN "videos"
                    WHEN mime_type IN ("application/pdf", "application/msword", "application/vnd.openxmlformats-officedocument.wordprocessingml.document", "application/vnd.ms-excel", "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet", "text/plain") THEN "documents"
                    ELSE "other"
                END as type,
                COUNT(*) as count,
                SUM(size) as total_size
            ')
            ->groupBy('type')
            ->get()
            ->keyBy('type')
            ->toArray();

        return [
            'categories' => [
                'total' => $totalCategories,
                'active' => $activeCategories,
                'inactive' => $totalCategories - $activeCategories
            ],
            'subcategories' => [
                'total' => $totalSubcategories,
                'active' => $activeSubcategories,
                'inactive' => $totalSubcategories - $activeSubcategories
            ],
            'courses' => [
                'total' => $totalCourses,
                'active' => $activeCourses,
                'inactive' => $totalCourses - $activeCourses,
                'featured' => $featuredCourses,
                'new_this_period' => $newCourses
            ],
            'modules' => [
                'total' => $totalModules,
                'active' => $activeModules,
                'inactive' => $totalModules - $activeModules,
                'new_this_period' => $newModules
            ],
            'news' => [
                'total' => $totalNews,
                'published' => $publishedNews,
                'draft' => $draftNews,
                'featured' => $featuredNews,
                'new_this_period' => $newArticles,
                'publication_rate' => $totalNews > 0 ? round(($publishedNews / $totalNews) * 100, 1) : 0
            ],
            'banners' => [
                'total' => $totalBanners,
                'active' => $activeBanners,
                'inactive' => $totalBanners - $activeBanners,
                'new_this_period' => $newBanners
            ],
            'media' => [
                'total_files' => $totalMedia,
                'public_files' => $publicMedia,
                'private_files' => $totalMedia - $publicMedia,
                'total_size' => $totalMediaSize,
                'total_size_formatted' => $this->formatBytes($totalMediaSize),
                'new_this_period' => $newMedia,
                'by_type' => $mediaByType
            ]
        ];
    }

    /**
     * Get engagement statistics
     */
    private function getEngagementStatistics(array $dateRange): array
    {
        // Banner engagement
        $totalBannerViews = Banner::sum('view_count');
        $totalBannerClicks = Banner::sum('click_count');
        $bannerCTR = $totalBannerViews > 0 ? round(($totalBannerClicks / $totalBannerViews) * 100, 2) : 0;

        // News engagement
        $totalNewsViews = News::sum('view_count');
        $avgNewsViews = News::where('is_published', true)->avg('view_count') ?? 0;

        // Top performing content
        $topBanners = Banner::orderBy('view_count', 'desc')
                           ->limit(5)
                           ->get(['id', 'title', 'view_count', 'click_count'])
                           ->map(function ($banner) {
                               return [
                                   'id' => $banner->id,
                                   'title' => $banner->title,
                                   'views' => $banner->view_count,
                                   'clicks' => $banner->click_count,
                                   'ctr' => $banner->view_count > 0 ? round(($banner->click_count / $banner->view_count) * 100, 2) : 0
                               ];
                           });

        $topNews = News::where('is_published', true)
                      ->orderBy('view_count', 'desc')
                      ->limit(5)
                      ->get(['id', 'title', 'view_count', 'published_at'])
                      ->map(function ($article) {
                          return [
                              'id' => $article->id,
                              'title' => $article->title,
                              'views' => $article->view_count,
                              'published_at' => $article->published_at->toISOString()
                          ];
                      });

        return [
            'banners' => [
                'total_views' => $totalBannerViews,
                'total_clicks' => $totalBannerClicks,
                'average_ctr' => $bannerCTR,
                'top_performing' => $topBanners
            ],
            'news' => [
                'total_views' => $totalNewsViews,
                'average_views' => round($avgNewsViews, 0),
                'top_performing' => $topNews
            ]
        ];
    }

    /**
     * Get system statistics
     */
    private function getSystemStatistics(): array
    {
        $totalSettings = Setting::count();
        $publicSettings = Setting::where('is_public', true)->count();
        
        $settingsByGroup = Setting::selectRaw('`group`, COUNT(*) as count')
                                 ->groupBy('group')
                                 ->get()
                                 ->pluck('count', 'group')
                                 ->toArray();

        return [
            'settings' => [
                'total' => $totalSettings,
                'public' => $publicSettings,
                'private' => $totalSettings - $publicSettings,
                'by_group' => $settingsByGroup
            ],
            'database' => [
                'total_tables' => 17, // Based on migrations
                'total_records' => $this->getTotalRecords()
            ],
            'storage' => [
                'media_files' => Media::count(),
                'total_size' => $this->formatBytes(Media::sum('size'))
            ]
        ];
    }

    /**
     * Get recent activity
     */
    private function getRecentActivity(): array
    {
        $activities = [];

        // Recent users
        $recentUsers = User::orderBy('created_at', 'desc')
                          ->limit(5)
                          ->get(['id', 'name', 'email', 'created_at'])
                          ->map(function ($user) {
                              return [
                                  'type' => 'user_registration',
                                  'title' => 'New user registered',
                                  'description' => $user->name . ' (' . $user->email . ')',
                                  'timestamp' => $user->created_at->toISOString(),
                                  'icon' => 'user-plus'
                              ];
                          });

        // Recent news
        $recentNews = News::orderBy('created_at', 'desc')
                         ->limit(3)
                         ->get(['id', 'title', 'is_published', 'created_at'])
                         ->map(function ($article) {
                             return [
                                 'type' => 'news_article',
                                 'title' => $article->is_published ? 'News article published' : 'News article created',
                                 'description' => $article->title,
                                 'timestamp' => $article->created_at->toISOString(),
                                 'icon' => 'newspaper'
                             ];
                         });

        // Recent media uploads
        $recentMedia = Media::orderBy('created_at', 'desc')
                           ->limit(3)
                           ->get(['id', 'title', 'filename', 'mime_type', 'created_at'])
                           ->map(function ($media) {
                               return [
                                   'type' => 'media_upload',
                                   'title' => 'Media file uploaded',
                                   'description' => $media->title ?: $media->filename,
                                   'timestamp' => $media->created_at->toISOString(),
                                   'icon' => str_starts_with($media->mime_type, 'image/') ? 'photo' : 'document'
                               ];
                           });

        return collect($activities)
            ->merge($recentUsers)
            ->merge($recentNews)
            ->merge($recentMedia)
            ->sortByDesc('timestamp')
            ->take(10)
            ->values()
            ->toArray();
    }

    /**
     * Get chart data for dashboard
     */
    private function getChartData(string $period): array
    {
        $days = $this->getPeriodDays($period);
        
        // User registrations over time
        $userRegistrations = User::selectRaw('DATE(created_at) as date, COUNT(*) as count')
                                ->where('created_at', '>=', now()->subDays($days))
                                ->groupBy('date')
                                ->orderBy('date')
                                ->get()
                                ->pluck('count', 'date')
                                ->toArray();

        // News articles over time
        $newsPublications = News::selectRaw('DATE(created_at) as date, COUNT(*) as count')
                               ->where('created_at', '>=', now()->subDays($days))
                               ->where('is_published', true)
                               ->groupBy('date')
                               ->orderBy('date')
                               ->get()
                               ->pluck('count', 'date')
                               ->toArray();

        // Fill missing dates with zeros
        $dateRange = [];
        for ($i = $days - 1; $i >= 0; $i--) {
            $date = now()->subDays($i)->format('Y-m-d');
            $dateRange[] = $date;
        }

        $userChartData = [];
        $newsChartData = [];
        
        foreach ($dateRange as $date) {
            $userChartData[] = [
                'date' => $date,
                'count' => $userRegistrations[$date] ?? 0
            ];
            $newsChartData[] = [
                'date' => $date,
                'count' => $newsPublications[$date] ?? 0
            ];
        }

        return [
            'user_registrations' => $userChartData,
            'news_publications' => $newsChartData,
            'period_days' => $days
        ];
    }

    /**
     * Get quick actions for admin
     */
    private function getQuickActions(): array
    {
        return [
            [
                'title' => 'Add New Category',
                'description' => 'Create a new course category',
                'icon' => 'folder-plus',
                'action' => 'create_category',
                'url' => '/admin/categories/create'
            ],
            [
                'title' => 'Publish News Article',
                'description' => 'Create and publish news',
                'icon' => 'newspaper',
                'action' => 'create_news',
                'url' => '/admin/news/create'
            ],
            [
                'title' => 'Upload Media',
                'description' => 'Upload images, videos, documents',
                'icon' => 'cloud-upload',
                'action' => 'upload_media',
                'url' => '/admin/media/upload'
            ],
            [
                'title' => 'Create Banner',
                'description' => 'Design promotional banner',
                'icon' => 'rectangle-stack',
                'action' => 'create_banner',
                'url' => '/admin/banners/create'
            ],
            [
                'title' => 'System Settings',
                'description' => 'Configure platform settings',
                'icon' => 'cog-6-tooth',
                'action' => 'manage_settings',
                'url' => '/admin/settings'
            ],
            [
                'title' => 'View Analytics',
                'description' => 'Detailed analytics dashboard',
                'icon' => 'chart-bar',
                'action' => 'view_analytics',
                'url' => '/admin/analytics'
            ]
        ];
    }

    /**
     * Get system alerts and notifications
     */
    private function getSystemAlerts(): array
    {
        $alerts = [];

        // Check for unverified users
        $unverifiedUsers = User::whereNull('email_verified_at')->count();
        if ($unverifiedUsers > 0) {
            $alerts[] = [
                'type' => 'warning',
                'title' => 'Unverified Users',
                'message' => "{$unverifiedUsers} users haven't verified their email addresses",
                'action' => 'View Users',
                'url' => '/admin/users?filter=unverified'
            ];
        }

        // Check for draft news articles
        $draftNews = News::where('is_published', false)->count();
        if ($draftNews > 0) {
            $alerts[] = [
                'type' => 'info',
                'title' => 'Draft Articles',
                'message' => "{$draftNews} news articles are waiting to be published",
                'action' => 'Review Drafts',
                'url' => '/admin/news?status=draft'
            ];
        }

        // Check for inactive banners
        $inactiveBanners = Banner::where('is_active', false)->count();
        if ($inactiveBanners > 0) {
            $alerts[] = [
                'type' => 'info',
                'title' => 'Inactive Banners',
                'message' => "{$inactiveBanners} banners are currently inactive",
                'action' => 'Manage Banners',
                'url' => '/admin/banners?status=inactive'
            ];
        }

        // Check storage usage (if over 80% of some limit)
        $totalMediaSize = Media::sum('size');
        $maxSize = 1073741824; // 1GB limit example
        if ($totalMediaSize > ($maxSize * 0.8)) {
            $alerts[] = [
                'type' => 'warning',
                'title' => 'Storage Usage High',
                'message' => 'Media storage is over 80% capacity',
                'action' => 'Manage Media',
                'url' => '/admin/media'
            ];
        }

        return $alerts;
    }

    /**
     * Helper methods
     */
    private function getDateRange(string $period): array
    {
        switch ($period) {
            case '7_days':
                return [now()->subDays(7), now()];
            case '30_days':
                return [now()->subDays(30), now()];
            case '90_days':
                return [now()->subDays(90), now()];
            case '1_year':
                return [now()->subYear(), now()];
            default:
                return [now()->subDays(30), now()];
        }
    }

    private function getPeriodDays(string $period): int
    {
        switch ($period) {
            case '7_days':
                return 7;
            case '30_days':
                return 30;
            case '90_days':
                return 90;
            case '1_year':
                return 365;
            default:
                return 30;
        }
    }

    private function calculateGrowthRate(string $table, array $dateRange): float
    {
        // Simple growth rate calculation
        // This would be more sophisticated in a real implementation
        return 12.5; // Placeholder
    }

    private function formatBytes(int $bytes): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }

    private function getTotalRecords(): int
    {
        return User::count() + 
               Category::count() + 
               Subcategory::count() + 
               Course::count() +
               CourseModule::count() +
               Banner::count() + 
               News::count() + 
               Setting::count() + 
               Media::count();
    }
}