<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\CourseModule;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Storage;

class CourseModuleController extends Controller
{
    /**
     * Get all modules for a specific course
     */
    public function index(Request $request, string $courseId): JsonResponse
    {
        try {
            $course = Course::find($courseId);
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $modules = CourseModule::where('course_id', $courseId)
                ->orderBy('order_index')
                ->get()
                ->map(function ($module) {
                    return [
                        'id' => $module->id,
                        'title' => $module->title,
                        'title_sw' => $module->title_sw,
                        'description' => $module->description,
                        'description_sw' => $module->description_sw,
                        'order_index' => $module->order_index,
                        'duration_minutes' => $module->duration_minutes,
                        'video_url' => $module->video_url,
                        'youtube_url' => $module->youtube_url,
                        'pdf_file_path' => $module->pdf_file_path,
                        'pdf_file_name' => $module->pdf_file_name,
                        'pdf_file_size' => $module->pdf_file_size,
                        'is_free_preview' => $module->is_free_preview,
                        'is_active' => $module->is_active,
                        'learning_objectives' => $module->learning_objectives,
                        'resources' => $module->resources,
                        'created_at' => $module->created_at->toISOString(),
                        'updated_at' => $module->updated_at->toISOString()
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'course' => [
                        'id' => $course->id,
                        'title' => $course->title
                    ],
                    'modules' => $modules,
                    'total_modules' => $modules->count()
                ],
                'message' => 'Course modules retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch course modules',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Create a new module for a course
     */
    public function store(Request $request, string $courseId): JsonResponse
    {
        try {
            $course = Course::find($courseId);
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $validated = $request->validate([
                'title' => 'required|string|max:255',
                'title_sw' => 'nullable|string|max:255',
                'description' => 'nullable|string',
                'description_sw' => 'nullable|string',
                'order_index' => 'nullable|integer|min:0',
                'duration_minutes' => 'nullable|integer|min:0',
                'video_url' => 'nullable|url',
                'youtube_url' => 'nullable|url',
                'is_free_preview' => 'nullable|boolean',
                'is_active' => 'nullable|boolean',
                'learning_objectives' => 'nullable|string',
                'resources' => 'nullable|array',
                'pdf_file' => 'nullable|file|mimes:pdf|max:10240'
            ]);

            // Set course_id
            $validated['course_id'] = $courseId;

            // Set default order_index if not provided
            if (!isset($validated['order_index'])) {
                $maxOrder = CourseModule::where('course_id', $courseId)->max('order_index') ?? 0;
                $validated['order_index'] = $maxOrder + 1;
            }

            // Set defaults
            $validated['is_free_preview'] = $validated['is_free_preview'] ?? false;
            $validated['is_active'] = $validated['is_active'] ?? true;

            // Handle PDF file upload
            if ($request->hasFile('pdf_file')) {
                $file = $request->file('pdf_file');
                $fileName = time() . '_' . $file->getClientOriginalName();
                $filePath = $file->storeAs('course-modules/pdfs', $fileName, 'public');
                
                $validated['pdf_file_path'] = $filePath;
                $validated['pdf_file_name'] = $file->getClientOriginalName();
                $validated['pdf_file_size'] = $file->getSize();
            }

            $module = CourseModule::create($validated);

            return response()->json([
                'success' => true,
                'data' => [
                    'module' => [
                        'id' => $module->id,
                        'title' => $module->title,
                        'title_sw' => $module->title_sw,
                        'description' => $module->description,
                        'description_sw' => $module->description_sw,
                        'order_index' => $module->order_index,
                        'duration_minutes' => $module->duration_minutes,
                        'video_url' => $module->video_url,
                        'youtube_url' => $module->youtube_url,
                        'pdf_file_path' => $module->pdf_file_path,
                        'pdf_file_name' => $module->pdf_file_name,
                        'pdf_file_size' => $module->pdf_file_size,
                        'is_free_preview' => $module->is_free_preview,
                        'is_active' => $module->is_active,
                        'learning_objectives' => $module->learning_objectives,
                        'resources' => $module->resources,
                        'created_at' => $module->created_at->toISOString(),
                        'updated_at' => $module->updated_at->toISOString()
                    ]
                ],
                'message' => 'Course module created successfully'
            ], 201);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create course module',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Show a specific module
     */
    public function show(string $courseId, string $moduleId): JsonResponse
    {
        try {
            $course = Course::find($courseId);
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $module = CourseModule::where('course_id', $courseId)->find($moduleId);
            if (!$module) {
                return response()->json([
                    'success' => false,
                    'message' => 'Module not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'module' => [
                        'id' => $module->id,
                        'course_id' => $module->course_id,
                        'title' => $module->title,
                        'title_sw' => $module->title_sw,
                        'description' => $module->description,
                        'description_sw' => $module->description_sw,
                        'order_index' => $module->order_index,
                        'duration_minutes' => $module->duration_minutes,
                        'video_url' => $module->video_url,
                        'youtube_url' => $module->youtube_url,
                        'pdf_file_path' => $module->pdf_file_path,
                        'pdf_file_name' => $module->pdf_file_name,
                        'pdf_file_size' => $module->pdf_file_size,
                        'is_free_preview' => $module->is_free_preview,
                        'is_active' => $module->is_active,
                        'learning_objectives' => $module->learning_objectives,
                        'resources' => $module->resources,
                        'created_at' => $module->created_at->toISOString(),
                        'updated_at' => $module->updated_at->toISOString()
                    ]
                ],
                'message' => 'Course module retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch course module',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Update a module
     */
    public function update(Request $request, string $courseId, string $moduleId): JsonResponse
    {
        try {
            $course = Course::find($courseId);
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $module = CourseModule::where('course_id', $courseId)->find($moduleId);
            if (!$module) {
                return response()->json([
                    'success' => false,
                    'message' => 'Module not found'
                ], 404);
            }

            $validated = $request->validate([
                'title' => 'sometimes|string|max:255',
                'title_sw' => 'sometimes|nullable|string|max:255',
                'description' => 'sometimes|nullable|string',
                'description_sw' => 'sometimes|nullable|string',
                'order_index' => 'sometimes|integer|min:0',
                'duration_minutes' => 'sometimes|nullable|integer|min:0',
                'video_url' => 'sometimes|nullable|url',
                'youtube_url' => 'sometimes|nullable|url',
                'is_free_preview' => 'sometimes|boolean',
                'is_active' => 'sometimes|boolean',
                'learning_objectives' => 'sometimes|nullable|string',
                'resources' => 'sometimes|nullable|array',
                'pdf_file' => 'sometimes|nullable|file|mimes:pdf|max:10240'
            ]);

            // Handle PDF file upload
            if ($request->hasFile('pdf_file')) {
                // Delete old file if exists
                if ($module->pdf_file_path && Storage::disk('public')->exists($module->pdf_file_path)) {
                    Storage::disk('public')->delete($module->pdf_file_path);
                }

                $file = $request->file('pdf_file');
                $fileName = time() . '_' . $file->getClientOriginalName();
                $filePath = $file->storeAs('course-modules/pdfs', $fileName, 'public');
                
                $validated['pdf_file_path'] = $filePath;
                $validated['pdf_file_name'] = $file->getClientOriginalName();
                $validated['pdf_file_size'] = $file->getSize();
            }

            $module->update($validated);

            return response()->json([
                'success' => true,
                'data' => [
                    'module' => [
                        'id' => $module->id,
                        'course_id' => $module->course_id,
                        'title' => $module->title,
                        'title_sw' => $module->title_sw,
                        'description' => $module->description,
                        'description_sw' => $module->description_sw,
                        'order_index' => $module->order_index,
                        'duration_minutes' => $module->duration_minutes,
                        'video_url' => $module->video_url,
                        'youtube_url' => $module->youtube_url,
                        'pdf_file_path' => $module->pdf_file_path,
                        'pdf_file_name' => $module->pdf_file_name,
                        'pdf_file_size' => $module->pdf_file_size,
                        'is_free_preview' => $module->is_free_preview,
                        'is_active' => $module->is_active,
                        'learning_objectives' => $module->learning_objectives,
                        'resources' => $module->resources,
                        'created_at' => $module->created_at->toISOString(),
                        'updated_at' => $module->updated_at->toISOString()
                    ]
                ],
                'message' => 'Course module updated successfully'
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update course module',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Delete a module
     */
    public function destroy(string $courseId, string $moduleId): JsonResponse
    {
        try {
            $course = Course::find($courseId);
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $module = CourseModule::where('course_id', $courseId)->find($moduleId);
            if (!$module) {
                return response()->json([
                    'success' => false,
                    'message' => 'Module not found'
                ], 404);
            }

            // Delete associated PDF file if exists
            if ($module->pdf_file_path && Storage::disk('public')->exists($module->pdf_file_path)) {
                Storage::disk('public')->delete($module->pdf_file_path);
            }

            $moduleTitle = $module->title;
            $module->delete();

            return response()->json([
                'success' => true,
                'message' => 'Course module deleted successfully',
                'data' => [
                    'deleted_module_id' => $moduleId,
                    'deleted_module_title' => $moduleTitle,
                    'deleted_at' => now()->toISOString()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete course module',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Reorder modules
     */
    public function reorder(Request $request, string $courseId): JsonResponse
    {
        try {
            $course = Course::find($courseId);
            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $validated = $request->validate([
                'modules' => 'required|array',
                'modules.*.id' => 'required|integer|exists:course_modules,id',
                'modules.*.order_index' => 'required|integer|min:0'
            ]);

            foreach ($validated['modules'] as $moduleData) {
                CourseModule::where('id', $moduleData['id'])
                    ->where('course_id', $courseId)
                    ->update(['order_index' => $moduleData['order_index']]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Course modules reordered successfully'
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to reorder course modules',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }
}