<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\Category;
use App\Models\Subcategory;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Str;

class CourseController extends Controller
{
    /**
     * List all courses with admin details
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'search' => 'nullable|string|max:255',
                'category_id' => 'nullable|string|exists:categories,id',
                'subcategory_id' => 'nullable|string|exists:subcategories,id',
                'status' => 'nullable|in:active,inactive',
                'is_active' => 'nullable|boolean',
                'is_featured' => 'nullable|boolean',
                'instructor_id' => 'nullable|integer|exists:users,id',
                'page' => 'nullable|integer|min:1',
                'per_page' => 'nullable|integer|min:1|max:100',
                'sort_by' => 'nullable|in:created_at,updated_at,title,enrollment_count,rating',
                'sort_order' => 'nullable|in:asc,desc'
            ]);

            $query = Course::with(['category', 'subcategory', 'instructor']);

            // Apply search filter
            if ($validated['search'] ?? null) {
                $search = $validated['search'];
                $query->where(function($q) use ($search) {
                    $q->where('title', 'like', "%{$search}%")
                      ->orWhere('title_sw', 'like', "%{$search}%")
                      ->orWhere('description', 'like', "%{$search}%")
                      ->orWhere('description_sw', 'like', "%{$search}%");
                });
            }

            // Apply filters
            if ($validated['category_id'] ?? null) {
                $query->where('category_id', $validated['category_id']);
            }

            if ($validated['subcategory_id'] ?? null) {
                $query->where('subcategory_id', $validated['subcategory_id']);
            }

            if ($validated['status'] ?? null) {
                $query->where('is_active', $validated['status'] === 'active');
            }

            if (isset($validated['is_active'])) {
                $query->where('is_active', $validated['is_active']);
            }

            if (isset($validated['is_featured'])) {
                $query->where('is_featured', $validated['is_featured']);
            }

            if ($validated['instructor_id'] ?? null) {
                $query->where('instructor_id', $validated['instructor_id']);
            }

            // Apply sorting
            $sortBy = $validated['sort_by'] ?? 'created_at';
            $sortOrder = $validated['sort_order'] ?? 'desc';
            $query->orderBy($sortBy, $sortOrder);

            // Pagination
            $perPage = $validated['per_page'] ?? 20;
            $courses = $query->paginate($perPage);

            // Transform the data
            $transformedCourses = $courses->getCollection()->map(function ($course) {
                return [
                    'id' => $course->id,
                    'title' => $course->title,
                    'title_sw' => $course->title_sw,
                    'description' => $course->description,
                    'description_sw' => $course->description_sw,
                    'category' => $course->category ? [
                        'id' => $course->category->id,
                        'name' => $course->category->name,
                        'name_sw' => $course->category->name_sw
                    ] : null,
                    'subcategory' => $course->subcategory ? [
                        'id' => $course->subcategory->id,
                        'name' => $course->subcategory->name,
                        'name_sw' => $course->subcategory->name_sw
                    ] : null,
                    'instructor' => $course->instructor ? [
                        'id' => $course->instructor->id,
                        'name' => $course->instructor->full_name,
                        'email' => $course->instructor->email
                    ] : null,
                    'price' => $course->price,
                    'original_price' => $course->original_price,
                    'currency' => $course->currency,
                    'discount_percentage' => $course->discount_percentage,
                    'duration' => $course->duration,
                    'difficulty_level' => $course->level,
                    'language' => $course->language,
                    'thumbnail_url' => $course->image_url,
                    'video_url' => $course->preview_video_url,
                    'youtube_url' => $course->youtube_url,
                    'course_pdf_path' => $course->course_pdf_path,
                    'course_pdf_name' => $course->course_pdf_name,
                    'course_pdf_size' => $course->course_pdf_size,
                    'is_active' => $course->is_active,
                    'is_featured' => $course->is_featured,
                    'is_active' => $course->is_active,
                    'enrollment_count' => $course->enrollment_count,
                    'rating' => $course->rating_average,
                    'review_count' => $course->review_count,
                    'created_at' => $course->created_at->toISOString(),
                    'updated_at' => $course->updated_at->toISOString()
                ];
            });

            // Calculate summary statistics
            $totalCourses = Course::count();
            $publishedCourses = Course::where('is_active', true)->count(); // Using is_active instead of is_published
            $draftCourses = Course::where('is_active', false)->count();
            $totalEnrollments = Course::sum('enrollment_count');

            return response()->json([
                'success' => true,
                'data' => [
                    'courses' => $transformedCourses,
                    'pagination' => [
                        'current_page' => $courses->currentPage(),
                        'total_pages' => $courses->lastPage(),
                        'total_items' => $courses->total(),
                        'items_per_page' => $courses->perPage(),
                        'has_next' => $courses->hasMorePages(),
                        'has_prev' => $courses->currentPage() > 1
                    ],
                    'summary' => [
                        'total_courses' => $totalCourses,
                        'published_courses' => $publishedCourses,
                        'draft_courses' => $draftCourses,
                        'total_enrollments' => $totalEnrollments
                    ]
                ],
                'message' => 'Courses retrieved successfully'
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch courses',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Create new course
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                // Core required fields
                'title' => 'required|string|max:255',
                'description' => 'required|string',
                'category_id' => 'required|string|exists:categories,id',
                'subcategory_id' => 'required|string|exists:subcategories,id',
                
                // Optional fields
                'title_sw' => 'nullable|string|max:255',
                'description_sw' => 'nullable|string',
                'instructor_id' => 'nullable|integer|exists:users,id', // Optional - defaults to system admin
                'price' => 'nullable|numeric|min:0',
                'original_price' => 'nullable|numeric|min:0',
                'currency' => 'nullable|in:TZS,USD',
                'duration' => 'nullable|string|max:50',
                'level' => 'nullable|in:Beginner,Intermediate,Advanced',
                'language' => 'nullable|in:English,Swahili,Both',
                'image_url' => 'nullable|url',
                'preview_video_url' => 'nullable|url',
                'youtube_url' => 'nullable|url',
                'course_pdf' => 'nullable|file|mimes:pdf|max:10240',
                'is_featured' => 'nullable|boolean',
                'is_active' => 'nullable|boolean'
            ]);

            // Validate that subcategory belongs to the selected category
            $subcategory = \App\Models\Subcategory::find($validated['subcategory_id']);
            if ($subcategory && $subcategory->category_id !== $validated['category_id']) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'errors' => [
                        'subcategory_id' => ['The selected subcategory does not belong to the chosen category.']
                    ]
                ], 422);
            }

            // Set defaults for missing optional fields
            $defaults = [
                'price' => 0.00,
                'currency' => 'TZS',
                'level' => 'Beginner',
                'language' => 'English',
                'is_active' => true,
                'is_featured' => false,
                'enrollment_count' => 0,
                'rating_average' => 0.00,
                'review_count' => 0,
                'total_lessons' => 0,
                'total_hours' => 0,
                'completion_rate' => 0,
                'certificate_offered' => false,
                'has_quizzes' => false,
                'has_assignments' => false,
                'has_downloadable_resources' => false
            ];

            // Apply defaults for any missing fields
            foreach ($defaults as $key => $defaultValue) {
                if (!isset($validated[$key]) || $validated[$key] === null) {
                    $validated[$key] = $defaultValue;
                }
            }

            // Handle PDF file upload
            if ($request->hasFile('course_pdf')) {
                $file = $request->file('course_pdf');
                $fileName = time() . '_' . $file->getClientOriginalName();
                $filePath = $file->storeAs('courses/pdfs', $fileName, 'public');
                
                $validated['course_pdf_path'] = $filePath;
                $validated['course_pdf_name'] = $file->getClientOriginalName();
                $validated['course_pdf_size'] = $file->getSize();
            }

            // Set default instructor to system admin if not provided
            if (!isset($validated['instructor_id']) || $validated['instructor_id'] === null) {
                // Use the system admin (ID: 2) as default instructor
                $systemAdmin = \App\Models\User::where('email', 'admin@exporterspoint.com')->first();
                if ($systemAdmin) {
                    $validated['instructor_id'] = $systemAdmin->id;
                } else {
                    // Fallback: find any admin user
                    $adminUser = \App\Models\User::where('role', 'admin')->first();
                    $validated['instructor_id'] = $adminUser ? $adminUser->id : null;
                }
            }

            $course = Course::create($validated);
            $course->load(['category', 'subcategory', 'instructor']);

            return response()->json([
                'success' => true,
                'data' => [
                    'course' => [
                        'id' => $course->id,
                        'title' => $course->title,
                        'title_sw' => $course->title_sw,
                        'description' => $course->description,
                        'description_sw' => $course->description_sw,
                        'category' => $course->category ? [
                            'id' => $course->category->id,
                            'name' => $course->category->name,
                            'name_sw' => $course->category->name_sw
                        ] : null,
                        'subcategory' => $course->subcategory ? [
                            'id' => $course->subcategory->id,
                            'name' => $course->subcategory->name,
                            'name_sw' => $course->subcategory->name_sw
                        ] : null,
                        'instructor' => $course->instructor ? [
                            'id' => $course->instructor->id,
                            'name' => $course->instructor->full_name,
                            'email' => $course->instructor->email
                        ] : null,
                        'price' => $course->price,
                        'original_price' => $course->original_price,
                        'currency' => $course->currency,
                        'discount_percentage' => $course->discount_percentage,
                        'duration' => $course->duration,
                        'difficulty_level' => $course->level,
                        'language' => $course->language,
                        'thumbnail_url' => $course->image_url,
                        'video_url' => $course->preview_video_url,
                        'youtube_url' => $course->youtube_url,
                        'course_pdf_path' => $course->course_pdf_path,
                        'course_pdf_name' => $course->course_pdf_name,
                        'course_pdf_size' => $course->course_pdf_size,
                        'is_active' => $course->is_active,
                        'is_featured' => $course->is_featured,
                        'enrollment_count' => $course->enrollment_count,
                        'rating' => $course->rating_average,
                        'review_count' => $course->review_count,
                        'created_at' => $course->created_at->toISOString(),
                        'updated_at' => $course->updated_at->toISOString()
                    ]
                ],
                'message' => 'Course created successfully'
            ], 201);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create course',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * View course details (admin view)
     */
    public function show(string $courseId): JsonResponse
    {
        try {
            $course = Course::with(['category', 'subcategory', 'instructor', 'lessons', 'reviews', 'enrollments', 'modules'])
                ->find($courseId);

            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'course' => [
                        'id' => $course->id,
                        'title' => $course->title,
                        'title_sw' => $course->title_sw,
                        'description' => $course->description,
                        'description_sw' => $course->description_sw,
                        'category' => $course->category ? [
                            'id' => $course->category->id,
                            'name' => $course->category->name,
                            'name_sw' => $course->category->name_sw
                        ] : null,
                        'subcategory' => $course->subcategory ? [
                            'id' => $course->subcategory->id,
                            'name' => $course->subcategory->name,
                            'name_sw' => $course->subcategory->name_sw
                        ] : null,
                        'instructor' => $course->instructor ? [
                            'id' => $course->instructor->id,
                            'name' => $course->instructor->full_name,
                            'email' => $course->instructor->email
                        ] : null,
                        'price' => $course->price,
                        'original_price' => $course->original_price,
                        'currency' => $course->currency,
                        'discount_percentage' => $course->discount_percentage,
                        'duration' => $course->duration,
                        'difficulty_level' => $course->level,
                        'language' => $course->language,
                        'thumbnail_url' => $course->image_url,
                        'video_url' => $course->preview_video_url,
                        'youtube_url' => $course->youtube_url,
                        'course_pdf_path' => $course->course_pdf_path,
                        'course_pdf_name' => $course->course_pdf_name,
                        'course_pdf_size' => $course->course_pdf_size,
                        'is_active' => $course->is_active,
                        'is_featured' => $course->is_featured,
                        'enrollment_count' => $course->enrollment_count,
                        'rating' => $course->rating_average,
                        'review_count' => $course->review_count,
                        'lessons_count' => $course->lessons->count(),
                        'enrollments_count' => $course->enrollments->count(),
                        'created_at' => $course->created_at->toISOString(),
                        'updated_at' => $course->updated_at->toISOString()
                    ]
                ],
                'message' => 'Course retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch course details',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Update course details
     */
    public function update(Request $request, string $courseId): JsonResponse
    {
        try {
            $course = Course::find($courseId);

            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $validated = $request->validate([
                'title' => 'sometimes|string|max:255',
                'title_sw' => 'sometimes|nullable|string|max:255',
                'description' => 'sometimes|string',
                'description_sw' => 'sometimes|nullable|string',
                'category_id' => 'sometimes|string|exists:categories,id',
                'subcategory_id' => 'sometimes|nullable|string|exists:subcategories,id',
                'instructor_id' => 'sometimes|nullable|string|exists:users,id',
                'price' => 'sometimes|numeric|min:0',
                'original_price' => 'sometimes|nullable|numeric|min:0',
                'currency' => 'sometimes|in:TZS,USD',
                'duration' => 'sometimes|nullable|string|max:50',
                'level' => 'sometimes|in:Beginner,Intermediate,Advanced',
                'language' => 'sometimes|in:English,Swahili,Both',
                'image_url' => 'sometimes|nullable|url',
                'preview_video_url' => 'sometimes|nullable|url',
                'youtube_url' => 'sometimes|nullable|url',
                'course_pdf' => 'sometimes|nullable|file|mimes:pdf|max:10240',
                'is_featured' => 'sometimes|boolean',
                'is_active' => 'sometimes|boolean'
            ]);

            // Handle PDF file upload
            if ($request->hasFile('course_pdf')) {
                // Delete old file if exists
                if ($course->course_pdf_path && \Illuminate\Support\Facades\Storage::disk('public')->exists($course->course_pdf_path)) {
                    \Illuminate\Support\Facades\Storage::disk('public')->delete($course->course_pdf_path);
                }

                $file = $request->file('course_pdf');
                $fileName = time() . '_' . $file->getClientOriginalName();
                $filePath = $file->storeAs('courses/pdfs', $fileName, 'public');
                
                $validated['course_pdf_path'] = $filePath;
                $validated['course_pdf_name'] = $file->getClientOriginalName();
                $validated['course_pdf_size'] = $file->getSize();
            }

            $course->update($validated);
            $course->load(['category', 'subcategory', 'instructor', 'modules']);

            return response()->json([
                'success' => true,
                'data' => [
                    'course' => [
                        'id' => $course->id,
                        'title' => $course->title,
                        'title_sw' => $course->title_sw,
                        'description' => $course->description,
                        'description_sw' => $course->description_sw,
                        'category' => $course->category ? [
                            'id' => $course->category->id,
                            'name' => $course->category->name,
                            'name_sw' => $course->category->name_sw
                        ] : null,
                        'subcategory' => $course->subcategory ? [
                            'id' => $course->subcategory->id,
                            'name' => $course->subcategory->name,
                            'name_sw' => $course->subcategory->name_sw
                        ] : null,
                        'instructor' => $course->instructor ? [
                            'id' => $course->instructor->id,
                            'name' => $course->instructor->full_name,
                            'email' => $course->instructor->email
                        ] : null,
                        'price' => $course->price,
                        'original_price' => $course->original_price,
                        'currency' => $course->currency,
                        'discount_percentage' => $course->discount_percentage,
                        'duration' => $course->duration,
                        'difficulty_level' => $course->level,
                        'language' => $course->language,
                        'thumbnail_url' => $course->image_url,
                        'video_url' => $course->preview_video_url,
                        'youtube_url' => $course->youtube_url,
                        'course_pdf_path' => $course->course_pdf_path,
                        'course_pdf_name' => $course->course_pdf_name,
                        'course_pdf_size' => $course->course_pdf_size,
                        'is_active' => $course->is_active,
                        'is_featured' => $course->is_featured,
                        'enrollment_count' => $course->enrollment_count,
                        'rating' => $course->rating_average,
                        'review_count' => $course->review_count,
                        'created_at' => $course->created_at->toISOString(),
                        'updated_at' => $course->updated_at->toISOString()
                    ]
                ],
                'message' => 'Course updated successfully'
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update course',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Delete course
     */
    public function destroy(string $courseId): JsonResponse
    {
        try {
            $course = Course::find($courseId);

            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            // Check if course has enrollments
            if ($course->enrollments()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete course with existing enrollments'
                ], 422);
            }

            $courseTitle = $course->title;
            $course->delete();

            return response()->json([
                'success' => true,
                'message' => 'Course deleted successfully',
                'data' => [
                    'deleted_course_id' => $courseId,
                    'deleted_course_title' => $courseTitle,
                    'deleted_at' => now()->toISOString()
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete course',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Toggle course status (publish/unpublish)
     */
    public function toggleStatus(Request $request, string $courseId): JsonResponse
    {
        try {
            $course = Course::find($courseId);

            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $validated = $request->validate([
                'is_active' => 'required|boolean'
            ]);

            $course->update($validated);

            return response()->json([
                'success' => true,
                'data' => [
                    'course_id' => $course->id,
                    'is_active' => $course->is_active,
                    'status' => $course->is_active ? 'active' : 'inactive'
                ],
                'message' => 'Course status updated successfully'
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update course status',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Toggle course featured status
     */
    public function toggleFeatured(Request $request, string $courseId): JsonResponse
    {
        try {
            $course = Course::find($courseId);

            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }

            $validated = $request->validate([
                'is_featured' => 'required|boolean'
            ]);

            $course->update($validated);

            return response()->json([
                'success' => true,
                'data' => [
                    'course_id' => $course->id,
                    'is_featured' => $course->is_featured
                ],
                'message' => 'Course featured status updated successfully'
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update course featured status',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get course statistics
     */
    public function statistics(): JsonResponse
    {
        try {
            $stats = [
                'totals' => [
                    'courses' => Course::count(),
                    'published' => Course::where('is_active', true)->count(), // Using is_active since is_published may not exist
                    'draft' => Course::where('is_active', false)->count(),
                    'featured' => Course::where('is_featured', true)->count(),
                    'active' => Course::where('is_active', true)->count()
                ],
                'by_category' => [], // Temporarily empty to avoid relationship errors
                'by_difficulty' => Course::selectRaw('level, COUNT(*) as count') // Using 'level' instead of 'difficulty_level'
                    ->whereNotNull('level')
                    ->groupBy('level')
                    ->get()
                    ->pluck('count', 'level'),
                'by_language' => Course::selectRaw('language, COUNT(*) as count')
                    ->whereNotNull('language')
                    ->groupBy('language')
                    ->get()
                    ->pluck('count', 'language'),
                'engagement' => [
                    'total_enrollments' => Course::sum('enrollment_count') ?? 0,
                    'average_rating' => Course::where('rating_average', '>', 0)->avg('rating_average') ?? 0, // Using rating_average
                    'total_reviews' => Course::sum('review_count') ?? 0
                ]
            ];

            // Try to get category stats if Category model exists and has proper relationship
            try {
                if (class_exists('App\Models\Category')) {
                    $stats['by_category'] = \App\Models\Category::all()->map(function ($category) {
                        $coursesCount = Course::where('category_id', $category->id)->count();
                        return [
                            'category_name' => $category->name ?? 'Unknown',
                            'courses_count' => $coursesCount
                        ];
                    })->toArray();
                }
            } catch (\Exception $e) {
                $stats['by_category'] = [];
            }

            return response()->json([
                'success' => true,
                'data' => $stats,
                'message' => 'Course statistics retrieved successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve course statistics',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }
}