<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;

class CertificateController extends Controller
{
    /**
     * List all certificates with filters
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'status' => 'nullable|in:pending,approved,issued,rejected',
                'course' => 'nullable|string|max:100',
                'student' => 'nullable|string|max:100',
                'date_from' => 'nullable|date',
                'date_to' => 'nullable|date',
                'page' => 'nullable|integer|min:1',
                'limit' => 'nullable|integer|min:1|max:100'
            ]);

            // Mock certificates data
            $allCertificates = [
                [
                    'id' => 'cert_1',
                    'certificateNumber' => 'EP-2025-001',
                    'student' => [
                        'id' => 'user_1',
                        'name' => 'John Mwalimu',
                        'email' => 'john@example.com'
                    ],
                    'course' => [
                        'id' => 'course_export_basics',
                        'title' => 'Export Business Fundamentals',
                        'titleSw' => 'Misingi ya Biashara ya Uuzaji Nje'
                    ],
                    'status' => 'issued',
                    'finalScore' => 92,
                    'completionDate' => '2025-01-22T16:30:00Z',
                    'issuedDate' => '2025-01-23T10:00:00Z',
                    'expiryDate' => '2027-01-23T10:00:00Z',
                    'downloadUrl' => 'https://certificates.exporterspoint.co.tz/EP-2025-001.pdf',
                    'verificationCode' => 'EP001VRF2025',
                    'issuedBy' => 'admin_1',
                    'createdAt' => '2025-01-22T16:30:00Z'
                ],
                [
                    'id' => 'cert_2',
                    'certificateNumber' => 'EP-2025-002',
                    'student' => [
                        'id' => 'user_2',
                        'name' => 'Sarah Kimani',
                        'email' => 'sarah@example.com'
                    ],
                    'course' => [
                        'id' => 'course_digital_marketing',
                        'title' => 'Digital Marketing for Exporters',
                        'titleSw' => 'Uuzaji wa Kidijitali kwa Wauzaji Nje'
                    ],
                    'status' => 'pending',
                    'finalScore' => 87,
                    'completionDate' => '2025-01-23T14:15:00Z',
                    'issuedDate' => null,
                    'expiryDate' => null,
                    'downloadUrl' => null,
                    'verificationCode' => null,
                    'issuedBy' => null,
                    'createdAt' => '2025-01-23T14:15:00Z'
                ],
                [
                    'id' => 'cert_3',
                    'certificateNumber' => 'EP-2025-003',
                    'student' => [
                        'id' => 'user_3',
                        'name' => 'David Mbugua',
                        'email' => 'david@example.com'
                    ],
                    'course' => [
                        'id' => 'course_export_basics',
                        'title' => 'Export Business Fundamentals',
                        'titleSw' => 'Misingi ya Biashara ya Uuzaji Nje'
                    ],
                    'status' => 'approved',
                    'finalScore' => 95,
                    'completionDate' => '2025-01-21T11:20:00Z',
                    'issuedDate' => null,
                    'expiryDate' => null,
                    'downloadUrl' => null,
                    'verificationCode' => null,
                    'approvedBy' => 'admin_1',
                    'approvedAt' => '2025-01-23T09:00:00Z',
                    'createdAt' => '2025-01-21T11:20:00Z'
                ]
            ];

            // Apply filters
            $filteredCertificates = $allCertificates;

            if ($validated['status'] ?? null) {
                $filteredCertificates = array_filter($filteredCertificates, function($cert) use ($validated) {
                    return $cert['status'] === $validated['status'];
                });
            }

            if ($validated['course'] ?? null) {
                $filteredCertificates = array_filter($filteredCertificates, function($cert) use ($validated) {
                    return $cert['course']['id'] === $validated['course'];
                });
            }

            // Pagination
            $page = $validated['page'] ?? 1;
            $limit = $validated['limit'] ?? 20;
            $totalCertificates = count($filteredCertificates);
            $offset = ($page - 1) * $limit;
            $paginatedCertificates = array_slice($filteredCertificates, $offset, $limit);

            return response()->json([
                'success' => true,
                'data' => [
                    'certificates' => array_values($paginatedCertificates),
                    'pagination' => [
                        'currentPage' => $page,
                        'totalPages' => ceil($totalCertificates / $limit),
                        'totalItems' => $totalCertificates,
                        'itemsPerPage' => $limit,
                        'hasNext' => $page < ceil($totalCertificates / $limit),
                        'hasPrev' => $page > 1
                    ],
                    'summary' => [
                        'totalCertificates' => count($allCertificates),
                        'pendingCertificates' => count(array_filter($allCertificates, fn($c) => $c['status'] === 'pending')),
                        'issuedCertificates' => count(array_filter($allCertificates, fn($c) => $c['status'] === 'issued')),
                        'approvedCertificates' => count(array_filter($allCertificates, fn($c) => $c['status'] === 'approved'))
                    ]
                ]
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch certificates',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * View certificate details
     */
    public function show(string $certificateId): JsonResponse
    {
        try {
            // Mock certificate details
            $certificate = [
                'id' => $certificateId,
                'certificateNumber' => 'EP-2025-001',
                'student' => [
                    'id' => 'user_1',
                    'name' => 'John Mwalimu',
                    'email' => 'john@example.com',
                    'phone' => '+255712345678',
                    'profileImage' => 'https://ui-avatars.com/api/?name=John+Mwalimu&size=200&background=7C3AED&color=fff'
                ],
                'course' => [
                    'id' => 'course_export_basics',
                    'title' => 'Export Business Fundamentals',
                    'titleSw' => 'Misingi ya Biashara ya Uuzaji Nje',
                    'instructor' => 'Sekela Mwambuli',
                    'duration' => '4 weeks',
                    'totalLessons' => 12
                ],
                'status' => 'issued',
                'finalScore' => 92,
                'passingScore' => 70,
                'completionDate' => '2025-01-22T16:30:00Z',
                'issuedDate' => '2025-01-23T10:00:00Z',
                'expiryDate' => '2027-01-23T10:00:00Z',
                'downloadUrl' => 'https://certificates.exporterspoint.co.tz/EP-2025-001.pdf',
                'verificationCode' => 'EP001VRF2025',
                'verificationUrl' => 'https://verify.exporterspoint.co.tz/EP001VRF2025',
                'issuedBy' => [
                    'id' => 'admin_1',
                    'name' => 'System Administrator',
                    'email' => 'admin@exporterspoint.co.tz'
                ],
                'lessonScores' => [
                    ['lesson' => 'Introduction to Export Business', 'score' => 95],
                    ['lesson' => 'Market Research and Analysis', 'score' => 88],
                    ['lesson' => 'Export Documentation', 'score' => 93]
                ],
                'createdAt' => '2025-01-22T16:30:00Z',
                'template' => [
                    'id' => 'template_1',
                    'name' => 'Standard Certificate Template',
                    'format' => 'PDF',
                    'size' => 'A4'
                ]
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'certificate' => $certificate
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch certificate details',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Approve/Issue/Pend certificate status
     */
    public function updateStatus(Request $request, string $certificateId): JsonResponse
    {
        try {
            $validated = $request->validate([
                'status' => 'required|in:pending,approved,issued,rejected',
                'reason' => 'nullable|string|max:500',
                'notes' => 'nullable|string|max:1000'
            ]);

            $status = $validated['status'];
            $reason = $validated['reason'] ?? null;
            $notes = $validated['notes'] ?? null;

            // Mock status update
            $updatedCertificate = [
                'id' => $certificateId,
                'certificateNumber' => 'EP-2025-001',
                'status' => $status,
                'reason' => $reason,
                'notes' => $notes,
                'updatedAt' => now()->toISOString(),
                'updatedBy' => 'admin_1'
            ];

            // Add status-specific fields
            if ($status === 'issued') {
                $updatedCertificate['issuedDate'] = now()->toISOString();
                $updatedCertificate['expiryDate'] = now()->addYears(2)->toISOString();
                $updatedCertificate['downloadUrl'] = "https://certificates.exporterspoint.co.tz/{$certificateId}.pdf";
                $updatedCertificate['verificationCode'] = strtoupper(substr(md5($certificateId), 0, 12));
            }

            $message = match($status) {
                'approved' => 'Certificate approved successfully',
                'issued' => 'Certificate issued successfully',
                'pending' => 'Certificate status changed to pending',
                'rejected' => 'Certificate rejected'
            };

            return response()->json([
                'success' => true,
                'message' => $message,
                'data' => [
                    'certificate' => $updatedCertificate
                ]
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update certificate status',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Download certificate
     */
    public function download(string $certificateId): JsonResponse
    {
        try {
            // Mock certificate download - in real implementation, generate and return PDF
            $downloadInfo = [
                'certificateId' => $certificateId,
                'certificateNumber' => 'EP-2025-001',
                'downloadUrl' => "https://certificates.exporterspoint.co.tz/{$certificateId}.pdf",
                'filename' => "Certificate_EP-2025-001.pdf",
                'fileSize' => '2.5MB',
                'format' => 'PDF',
                'generatedAt' => now()->toISOString(),
                'expiresAt' => now()->addHours(24)->toISOString() // Download link expires in 24 hours
            ];

            return response()->json([
                'success' => true,
                'message' => 'Certificate download link generated',
                'data' => $downloadInfo
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to generate certificate download',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }
}