<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Banner;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;

class BannerController extends Controller
{
    /**
     * Display a listing of banners.
     */
    public function index(Request $request): JsonResponse
    {
        $query = Banner::query();

        // Filter by status
        if ($request->has('status')) {
            if ($request->status === 'active') {
                $query->where('is_active', true);
            } elseif ($request->status === 'inactive') {
                $query->where('is_active', false);
            }
        }

        // Filter by action type
        if ($request->has('action_type')) {
            $query->where('action_type', $request->action_type);
        }

        // Filter by date range
        if ($request->has('date_from')) {
            $query->where('start_date', '>=', $request->date_from);
        }
        if ($request->has('date_to')) {
            $query->where('end_date', '<=', $request->date_to);
        }

        // Search functionality
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('title_sw', 'like', "%{$search}%")
                  ->orWhere('subtitle', 'like', "%{$search}%")
                  ->orWhere('subtitle_sw', 'like', "%{$search}%")
                  ->orWhere('label', 'like', "%{$search}%")
                  ->orWhere('label_sw', 'like', "%{$search}%");
            });
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'order_index');
        $sortOrder = $request->get('sort_order', 'asc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->get('per_page', 15);
        $banners = $query->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $banners,
            'message' => 'Banners retrieved successfully'
        ]);
    }

    /**
     * Store a newly created banner.
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'subtitle' => 'required|string|max:255',
            'label' => 'required|string|max:100',
            'image_url' => 'required|string|max:500',
            'title_sw' => 'nullable|string|max:255',
            'subtitle_sw' => 'nullable|string|max:255',
            'label_sw' => 'nullable|string|max:100',
            'button_text' => 'nullable|string|max:50',
            'button_text_sw' => 'nullable|string|max:50',
            'background_color' => 'nullable|string|max:7|regex:/^#[0-9A-Fa-f]{6}$/',
            'text_color' => 'nullable|string|max:7|regex:/^#[0-9A-Fa-f]{6}$/',
            'action_type' => 'nullable|string|in:none,url,course,category,app_page',
            'action_target' => 'nullable|string|max:500',
            'order_index' => 'nullable|integer|min:0',
            'is_active' => 'nullable|boolean',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        // Generate unique ID
        $validated['id'] = Str::uuid();

        // Set default order_index if not provided
        if (!isset($validated['order_index'])) {
            $maxOrder = Banner::max('order_index') ?? 0;
            $validated['order_index'] = $maxOrder + 1;
        }

        // Set default values
        $validated['is_active'] = $validated['is_active'] ?? true;
        $validated['click_count'] = 0;
        $validated['view_count'] = 0;
        $validated['action_type'] = $validated['action_type'] ?? 'none';
        $validated['button_text'] = $validated['button_text'] ?? 'Learn More';
        $validated['button_text_sw'] = $validated['button_text_sw'] ?? 'Jifunze Zaidi';

        // Set default colors if not provided
        $validated['background_color'] = $validated['background_color'] ?? '#007bff';
        $validated['text_color'] = $validated['text_color'] ?? '#ffffff';

        $banner = Banner::create($validated);

        return response()->json([
            'success' => true,
            'data' => $banner,
            'message' => 'Banner created successfully'
        ], 201);
    }

    /**
     * Display the specified banner.
     */
    public function show(string $id): JsonResponse
    {
        $banner = Banner::find($id);

        if (!$banner) {
            return response()->json([
                'success' => false,
                'message' => 'Banner not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $banner,
            'message' => 'Banner retrieved successfully'
        ]);
    }

    /**
     * Update the specified banner.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        $banner = Banner::find($id);

        if (!$banner) {
            return response()->json([
                'success' => false,
                'message' => 'Banner not found'
            ], 404);
        }

        $validated = $request->validate([
            'title' => 'sometimes|required|string|max:255',
            'title_sw' => 'nullable|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'subtitle_sw' => 'nullable|string|max:255',
            'label' => 'nullable|string|max:100',
            'label_sw' => 'nullable|string|max:100',
            'button_text' => 'nullable|string|max:50',
            'button_text_sw' => 'nullable|string|max:50',
            'image_url' => 'nullable|string|max:500',
            'background_color' => 'nullable|string|max:7|regex:/^#[0-9A-Fa-f]{6}$/',
            'text_color' => 'nullable|string|max:7|regex:/^#[0-9A-Fa-f]{6}$/',
            'action_type' => 'sometimes|required|string|in:none,url,course,category,app_page',
            'action_target' => 'nullable|string|max:500',
            'order_index' => 'nullable|integer|min:0',
            'is_active' => 'boolean',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        $banner->update($validated);

        return response()->json([
            'success' => true,
            'data' => $banner,
            'message' => 'Banner updated successfully'
        ]);
    }

    /**
     * Remove the specified banner.
     */
    public function destroy(string $id): JsonResponse
    {
        $banner = Banner::find($id);

        if (!$banner) {
            return response()->json([
                'success' => false,
                'message' => 'Banner not found'
            ], 404);
        }

        $banner->delete();

        return response()->json([
            'success' => true,
            'message' => 'Banner deleted successfully'
        ]);
    }

    /**
     * Update banner status.
     */
    public function updateStatus(Request $request, string $id): JsonResponse
    {
        $banner = Banner::find($id);

        if (!$banner) {
            return response()->json([
                'success' => false,
                'message' => 'Banner not found'
            ], 404);
        }

        $validated = $request->validate([
            'is_active' => 'required|boolean'
        ]);

        $banner->update($validated);

        return response()->json([
            'success' => true,
            'data' => $banner,
            'message' => 'Banner status updated successfully'
        ]);
    }

    /**
     * Reorder banners.
     */
    public function reorder(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'banners' => 'required|array',
            'banners.*.id' => 'required|string|exists:banners,id',
            'banners.*.order_index' => 'required|integer|min:0'
        ]);

        foreach ($validated['banners'] as $bannerData) {
            Banner::where('id', $bannerData['id'])
                  ->update(['order_index' => $bannerData['order_index']]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Banners reordered successfully'
        ]);
    }

    /**
     * Get banner analytics.
     */
    public function analytics(string $id): JsonResponse
    {
        $banner = Banner::find($id);

        if (!$banner) {
            return response()->json([
                'success' => false,
                'message' => 'Banner not found'
            ], 404);
        }

        $analytics = [
            'banner_id' => $banner->id,
            'title' => $banner->title,
            'view_count' => $banner->view_count,
            'click_count' => $banner->click_count,
            'click_through_rate' => $banner->view_count > 0 
                ? round(($banner->click_count / $banner->view_count) * 100, 2) 
                : 0,
            'is_active' => $banner->is_active,
            'start_date' => $banner->start_date,
            'end_date' => $banner->end_date,
            'created_at' => $banner->created_at,
        ];

        return response()->json([
            'success' => true,
            'data' => $analytics,
            'message' => 'Banner analytics retrieved successfully'
        ]);
    }

    /**
     * Get banner statistics.
     */
    public function statistics(): JsonResponse
    {
        $stats = [
            'total_banners' => Banner::count(),
            'active_banners' => Banner::where('is_active', true)->count(),
            'inactive_banners' => Banner::where('is_active', false)->count(),
            'scheduled_banners' => Banner::where('start_date', '>', now())->count(),
            'expired_banners' => Banner::where('end_date', '<', now())->count(),
            'total_views' => Banner::sum('view_count'),
            'total_clicks' => Banner::sum('click_count'),
            'average_ctr' => Banner::selectRaw('AVG(CASE WHEN view_count > 0 THEN (click_count / view_count) * 100 ELSE 0 END) as avg_ctr')
                                  ->first()->avg_ctr ?? 0,
            'banners_by_action_type' => Banner::selectRaw('action_type, COUNT(*) as count')
                                             ->groupBy('action_type')
                                             ->get()
                                             ->pluck('count', 'action_type')
                                             ->toArray()
        ];

        return response()->json([
            'success' => true,
            'data' => $stats,
            'message' => 'Banner statistics retrieved successfully'
        ]);
    }

    /**
     * Duplicate a banner.
     */
    public function duplicate(string $id): JsonResponse
    {
        $banner = Banner::find($id);

        if (!$banner) {
            return response()->json([
                'success' => false,
                'message' => 'Banner not found'
            ], 404);
        }

        $newBanner = $banner->replicate();
        $newBanner->id = Str::uuid();
        $newBanner->title = $banner->title . ' (Copy)';
        $newBanner->is_active = false;
        $newBanner->click_count = 0;
        $newBanner->view_count = 0;
        $newBanner->order_index = (Banner::max('order_index') ?? 0) + 1;
        $newBanner->save();

        return response()->json([
            'success' => true,
            'data' => $newBanner,
            'message' => 'Banner duplicated successfully'
        ], 201);
    }
}