<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    /**
     * Admin login
     */
    public function login(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'email' => 'required|email',
                'password' => 'required|string|min:6',
                'remember' => 'nullable|boolean'
            ]);

            // Mock admin authentication - in real implementation, check against admins table
            $mockAdmins = [
                [
                    'id' => 'admin_1',
                    'email' => 'admin@exporterspoint.co.tz',
                    'password' => Hash::make('admin123'),
                    'name' => 'System Administrator',
                    'role' => 'super_admin',
                    'permissions' => ['all'],
                    'isActive' => true
                ],
                [
                    'id' => 'admin_2',
                    'email' => 'manager@exporterspoint.co.tz',
                    'password' => Hash::make('manager123'),
                    'name' => 'Content Manager',
                    'role' => 'content_manager',
                    'permissions' => ['courses', 'news', 'users'],
                    'isActive' => true
                ]
            ];

            $admin = collect($mockAdmins)->firstWhere('email', $validated['email']);

            if (!$admin || !Hash::check($validated['password'], $admin['password'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid credentials'
                ], 401);
            }

            if (!$admin['isActive']) {
                return response()->json([
                    'success' => false,
                    'message' => 'Account is deactivated'
                ], 403);
            }

            // Generate session token (in real implementation, use Laravel Sanctum)
            $token = base64_encode(json_encode([
                'admin_id' => $admin['id'],
                'email' => $admin['email'],
                'expires_at' => now()->addHours(8)->timestamp
            ]));

            return response()->json([
                'success' => true,
                'message' => 'Login successful',
                'data' => [
                    'admin' => [
                        'id' => $admin['id'],
                        'name' => $admin['name'],
                        'email' => $admin['email'],
                        'role' => $admin['role'],
                        'permissions' => $admin['permissions']
                    ],
                    'token' => $token,
                    'expiresAt' => now()->addHours(8)->toISOString()
                ]
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Login failed',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Admin logout
     */
    public function logout(Request $request): JsonResponse
    {
        try {
            // In real implementation, invalidate token/session
            // For now, just return success response
            
            return response()->json([
                'success' => true,
                'message' => 'Logout successful'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Logout failed',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Get current admin profile
     */
    public function me(Request $request): JsonResponse
    {
        try {
            // Mock current admin based on token (in real implementation, get from auth middleware)
            $currentAdmin = [
                'id' => 'admin_1',
                'name' => 'System Administrator',
                'email' => 'admin@exporterspoint.co.tz',
                'role' => 'super_admin',
                'permissions' => ['all'],
                'avatar' => 'https://ui-avatars.com/api/?name=System+Administrator&size=200&background=7C3AED&color=fff',
                'lastLoginAt' => now()->subMinutes(30)->toISOString(),
                'createdAt' => '2025-01-01T00:00:00Z',
                'stats' => [
                    'sessionsToday' => 3,
                    'actionsToday' => 15,
                    'totalSessions' => 145
                ]
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'admin' => $currentAdmin
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch admin profile',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }
}