<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Validation\ValidationException;

class AnalyticsController extends Controller
{
    /**
     * Admin dashboard summary stats
     */
    public function dashboard(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'period' => 'nullable|in:today,week,month,quarter,year',
                'timezone' => 'nullable|string|max:50'
            ]);

            $period = $validated['period'] ?? 'month';

            // Mock dashboard statistics
            $dashboardStats = [
                'overview' => [
                    'totalUsers' => 1247,
                    'totalCourses' => 23,
                    'totalEnrollments' => 3456,
                    'totalRevenue' => 172800000, // TZS
                    'totalCertificates' => 287
                ],
                'growth' => [
                    'usersGrowth' => 12.5, // percentage
                    'enrollmentsGrowth' => 8.3,
                    'revenueGrowth' => 15.7,
                    'coursesGrowth' => 4.5
                ],
                'userStats' => [
                    'activeUsers' => 892,
                    'newRegistrations' => 45,
                    'verifiedUsers' => 1156,
                    'instructors' => 12,
                    'onlineNow' => 23
                ],
                'courseStats' => [
                    'publishedCourses' => 20,
                    'draftCourses' => 3,
                    'featuredCourses' => 6,
                    'averageRating' => 4.7,
                    'totalLessons' => 342
                ],
                'revenueStats' => [
                    'totalRevenue' => 172800000,
                    'monthlyRevenue' => 14400000,
                    'averageOrderValue' => 50000,
                    'topSellingCourse' => [
                        'id' => 'course_export_basics',
                        'title' => 'Export Business Fundamentals',
                        'revenue' => 17100000,
                        'enrollments' => 342
                    ]
                ],
                'recentActivity' => [
                    [
                        'type' => 'enrollment',
                        'description' => 'New enrollment in Export Business Fundamentals',
                        'user' => 'John Mwalimu',
                        'timestamp' => now()->subMinutes(5)->toISOString()
                    ],
                    [
                        'type' => 'course_completion',
                        'description' => 'Course completed: Digital Marketing for Exporters',
                        'user' => 'Sarah Kimani',
                        'timestamp' => now()->subMinutes(15)->toISOString()
                    ],
                    [
                        'type' => 'new_user',
                        'description' => 'New user registration',
                        'user' => 'David Mbugua',
                        'timestamp' => now()->subMinutes(30)->toISOString()
                    ]
                ],
                'chartData' => [
                    'enrollmentsChart' => [
                        'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul'],
                        'data' => [45, 67, 89, 123, 156, 178, 192]
                    ],
                    'revenueChart' => [
                        'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul'],
                        'data' => [2250000, 3350000, 4450000, 6150000, 7800000, 8900000, 9600000]
                    ]
                ]
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'dashboard' => $dashboardStats,
                    'period' => $period,
                    'generatedAt' => now()->toISOString()
                ]
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch dashboard analytics',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * User analytics
     */
    public function userAnalytics(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'period' => 'nullable|in:week,month,quarter,year',
                'segment' => 'nullable|in:all,new,active,inactive,instructors'
            ]);

            $period = $validated['period'] ?? 'month';
            $segment = $validated['segment'] ?? 'all';

            // Mock user analytics
            $userAnalytics = [
                'summary' => [
                    'totalUsers' => 1247,
                    'activeUsers' => 892,
                    'newUsers' => 45,
                    'churnRate' => 5.2, // percentage
                    'retentionRate' => 78.5,
                    'averageSessionDuration' => 1440 // minutes
                ],
                'demographics' => [
                    'byLanguage' => [
                        'Swahili' => 856,
                        'English' => 391
                    ],
                    'byAuthProvider' => [
                        'email' => 789,
                        'google' => 458
                    ],
                    'byRole' => [
                        'student' => 1235,
                        'instructor' => 12
                    ]
                ],
                'engagement' => [
                    'dailyActiveUsers' => 234,
                    'weeklyActiveUsers' => 567,
                    'monthlyActiveUsers' => 892,
                    'averageCoursesPerUser' => 2.8,
                    'averageCompletionRate' => 73.5
                ],
                'registrationTrends' => [
                    'labels' => ['Week 1', 'Week 2', 'Week 3', 'Week 4'],
                    'data' => [12, 18, 23, 15]
                ],
                'activityTrends' => [
                    'labels' => ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                    'data' => [145, 167, 189, 156, 134, 98, 87]
                ],
                'topUsers' => [
                    [
                        'id' => 'user_1',
                        'name' => 'John Mwalimu',
                        'coursesCompleted' => 5,
                        'totalLearningTime' => 7200, // minutes
                        'certificatesEarned' => 3
                    ],
                    [
                        'id' => 'user_2',
                        'name' => 'Sarah Kimani',
                        'coursesCompleted' => 4,
                        'totalLearningTime' => 5760,
                        'certificatesEarned' => 2
                    ]
                ]
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'userAnalytics' => $userAnalytics,
                    'period' => $period,
                    'segment' => $segment,
                    'generatedAt' => now()->toISOString()
                ]
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch user analytics',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Course analytics
     */
    public function courseAnalytics(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'period' => 'nullable|in:week,month,quarter,year',
                'course_id' => 'nullable|string|max:100'
            ]);

            $period = $validated['period'] ?? 'month';
            $courseId = $validated['course_id'] ?? null;

            // Mock course analytics
            $courseAnalytics = [
                'summary' => [
                    'totalCourses' => 23,
                    'publishedCourses' => 20,
                    'totalEnrollments' => 3456,
                    'totalCompletions' => 2534,
                    'averageCompletionRate' => 73.3,
                    'averageRating' => 4.7
                ],
                'performance' => [
                    'mostPopularCourse' => [
                        'id' => 'course_export_basics',
                        'title' => 'Export Business Fundamentals',
                        'enrollments' => 342,
                        'completionRate' => 87,
                        'rating' => 4.8
                    ],
                    'highestRatedCourse' => [
                        'id' => 'course_legal_aspects',
                        'title' => 'Legal Aspects of International Trade',
                        'rating' => 4.9,
                        'enrollments' => 267,
                        'completionRate' => 82
                    ],
                    'lowestPerformingCourse' => [
                        'id' => 'course_draft_logistics',
                        'title' => 'International Logistics Management',
                        'enrollments' => 23,
                        'completionRate' => 45,
                        'rating' => 3.8
                    ]
                ],
                'enrollmentTrends' => [
                    'labels' => ['Week 1', 'Week 2', 'Week 3', 'Week 4'],
                    'data' => [89, 123, 145, 167]
                ],
                'completionTrends' => [
                    'labels' => ['Week 1', 'Week 2', 'Week 3', 'Week 4'],
                    'data' => [67, 89, 98, 112]
                ],
                'categoryPerformance' => [
                    [
                        'category' => 'Export Business',
                        'courses' => 8,
                        'enrollments' => 1456,
                        'completionRate' => 78.5,
                        'avgRating' => 4.6
                    ],
                    [
                        'category' => 'Digital Marketing',
                        'courses' => 5,
                        'enrollments' => 789,
                        'completionRate' => 72.3,
                        'avgRating' => 4.4
                    ],
                    [
                        'category' => 'Legal Aspects',
                        'courses' => 4,
                        'enrollments' => 567,
                        'completionRate' => 81.2,
                        'avgRating' => 4.8
                    ]
                ],
                'instructorPerformance' => [
                    [
                        'instructor' => 'Sekela Mwambuli',
                        'courses' => 6,
                        'totalEnrollments' => 1234,
                        'avgRating' => 4.7,
                        'avgCompletionRate' => 79.5
                    ],
                    [
                        'instructor' => 'Dr. Amina Hassan',
                        'courses' => 4,
                        'totalEnrollments' => 889,
                        'avgRating' => 4.5,
                        'avgCompletionRate' => 75.2
                    ]
                ]
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'courseAnalytics' => $courseAnalytics,
                    'period' => $period,
                    'courseId' => $courseId,
                    'generatedAt' => now()->toISOString()
                ]
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch course analytics',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }

    /**
     * Revenue analytics
     */
    public function revenueAnalytics(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'period' => 'nullable|in:week,month,quarter,year',
                'currency' => 'nullable|in:TZS,USD'
            ]);

            $period = $validated['period'] ?? 'month';
            $currency = $validated['currency'] ?? 'TZS';

            // Mock revenue analytics
            $revenueAnalytics = [
                'summary' => [
                    'totalRevenue' => 172800000, // TZS
                    'monthlyRevenue' => 14400000,
                    'weeklyRevenue' => 3600000,
                    'dailyRevenue' => 514285,
                    'averageOrderValue' => 50000,
                    'totalOrders' => 3456,
                    'refunds' => 23400000
                ],
                'growth' => [
                    'revenueGrowth' => 15.7, // percentage
                    'orderGrowth' => 12.3,
                    'avgOrderValueGrowth' => 3.1
                ],
                'revenueByPeriod' => [
                    'labels' => ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul'],
                    'revenue' => [8200000, 9500000, 11200000, 13800000, 15600000, 16900000, 18400000],
                    'orders' => [164, 190, 224, 276, 312, 338, 368]
                ],
                'revenueByCategory' => [
                    [
                        'category' => 'Export Business',
                        'revenue' => 89400000,
                        'percentage' => 51.7,
                        'orders' => 1788
                    ],
                    [
                        'category' => 'Digital Marketing',
                        'revenue' => 45600000,
                        'percentage' => 26.4,
                        'orders' => 912
                    ],
                    [
                        'category' => 'Legal Aspects',
                        'revenue' => 28200000,
                        'percentage' => 16.3,
                        'orders' => 564
                    ],
                    [
                        'category' => 'Transportation',
                        'revenue' => 9600000,
                        'percentage' => 5.6,
                        'orders' => 192
                    ]
                ],
                'topSellingCourses' => [
                    [
                        'id' => 'course_export_basics',
                        'title' => 'Export Business Fundamentals',
                        'revenue' => 17100000,
                        'orders' => 342,
                        'avgOrderValue' => 50000
                    ],
                    [
                        'id' => 'course_digital_marketing',
                        'title' => 'Digital Marketing for Exporters',
                        'revenue' => 12870000,
                        'orders' => 198,
                        'avgOrderValue' => 65000
                    ],
                    [
                        'id' => 'course_legal_aspects',
                        'title' => 'Legal Aspects of International Trade',
                        'revenue' => 14685000,
                        'orders' => 267,
                        'avgOrderValue' => 55000
                    ]
                ],
                'paymentMethods' => [
                    [
                        'method' => 'Mobile Money',
                        'revenue' => 103680000,
                        'percentage' => 60.0,
                        'orders' => 2074
                    ],
                    [
                        'method' => 'Bank Transfer',
                        'revenue' => 51840000,
                        'percentage' => 30.0,
                        'orders' => 1037
                    ],
                    [
                        'method' => 'Credit Card',
                        'revenue' => 17280000,
                        'percentage' => 10.0,
                        'orders' => 345
                    ]
                ]
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'revenueAnalytics' => $revenueAnalytics,
                    'period' => $period,
                    'currency' => $currency,
                    'generatedAt' => now()->toISOString()
                ]
            ]);

        } catch (ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch revenue analytics',
                'error' => config('app.debug') ? $e->getMessage() : 'Internal server error'
            ], 500);
        }
    }
}